/*
 * serialpps-ppsapi-provider.c - derived from monolithic timepps.h
 *				 for serialpps.sys by Dave Hart
 */

/***********************************************************************
 *								       *
 * Copyright (c) David L. Mills 1999-2009			       *
 *								       *
 * Permission to use, copy, modify, and distribute this software and   *
 * its documentation for any purpose and without fee is hereby	       *
 * granted, provided that the above copyright notice appears in all    *
 * copies and that both the copyright notice and this permission       *
 * notice appear in supporting documentation, and that the name        *
 * University of Delaware not be used in advertising or publicity      *
 * pertaining to distribution of the software without specific,        *
 * written prior permission. The University of Delaware makes no       *
 * representations about the suitability this software for any	       *
 * purpose. It is provided "as is" without express or implied          *
 * warranty.							       *
 *								       *
 ***********************************************************************
 *								       *
 * This header file complies with "Pulse-Per-Second API for UNIX-like  *
 * Operating Systems, Version 1.0", rfc2783. Credit is due Jeff Mogul  *
 * and Marc Brett, from whom much of this code was shamelessly stolen. *
 *								       *
 * This serialpps-ppsapi-provider.dll implements the PPSAPI provider   *
 * for serialpps.sys, which is a very lightly patched Windows	       *
 * serial.sys with CD timestamping support.
 *								       *
 * This Windows version was derived by Dave Hart		       *
 * <davehart@davehart.com> from David L. Mills' timepps-Solaris.h      *
 *								       *
 ***********************************************************************
 *								       *
 * Some of this include file					       *
 * Copyright (c) 1999 by Ulrich Windl,				       *
 *	based on code by Reg Clemens <reg@dwf.com>		       *
 *		based on code by Poul-Henning Kamp <phk@FreeBSD.org>   *
 *								       *
 ***********************************************************************
 *								       *
 * "THE BEER-WARE LICENSE" (Revision 42):			       *
 * <phk@FreeBSD.org> wrote this file.  As long as you retain this      *
 * notice you can do whatever you want with this stuff. If we meet some*
 * day, and you think this stuff is worth it, you can buy me a beer    *
 * in return.	Poul-Henning Kamp				       *
 *								       *
 **********************************************************************/

/*
 * Implementation note: the logical states ``assert'' and ``clear''
 * are implemented in terms of the UART register, i.e. ``assert''
 * means the bit is set.
 * This is opposite of the RS-232 spec for the CD logical state.
 */


#define PPSAPI_PROVIDER_EXPORTS
#include "serialpps-ppsapi-provider.h"

pcreate_pps_handle		p_create_pps_handle;
ppps_ntp_timestamp_from_counter	p_ntp_timestamp_from_counter;

#define SERIALPPS_CAPS	(PPS_CAPTUREASSERT | PPS_OFFSETASSERT	\
			 | PPS_TSFMT_TSPEC | PPS_TSFMT_NTPFP)
#define SERIALPPS_RO	(PPS_CANWAIT | PPS_CANPOLL)

typedef struct _OLD_SERIAL_PPS_STAMPS {
	LARGE_INTEGER Timestamp;
	LARGE_INTEGER Counterstamp;
} OLD_SERIAL_PPS_STAMPS, *POLDSERIAL_PPS_STAMPS;

typedef struct _SERIAL_PPS_STAMPS {
	LARGE_INTEGER Timestamp;
	LARGE_INTEGER Counterstamp;
	DWORD SeqNum;
} SERIAL_PPS_STAMPS, *PSERIAL_PPS_STAMPS;

#define IOCTL_SERIAL_GET_PPS_STAMPS	\
	CTL_CODE(FILE_DEVICE_SERIAL_PORT,114,METHOD_BUFFERED,FILE_ANY_ACCESS)

/*
 * The ntp_timestamp_from_counter callback into timepps.h routines in
 * the host is saved in each unit separately, so that binaries that
 * inline timepps.h into multiple source files (such as refclock_atom.c
 * and a number of other ntpd refclocks including refclock_nmea.c) will
 * get called back in the correct instance for each unit.  This assumes 
 * that ppsapi_prov_init for subsequent instances happens only after the
 * first instance has completed all time_pps_create() calls it will
 * invoke, which is a safe assumption at least for ntpd.
 */
typedef struct serial_unit_tag {
	OVERLAPPED			ol;
	HANDLE				device;
	ppps_ntp_timestamp_from_counter	p_ntp_timestamp_from_counter;
	DWORD				prior_seq;
	ntp_fp_t			prior_stamp;
} serial_unit;


/*
 * DllMain - DLL entrypoint, no-op.
 */
BOOL APIENTRY DllMain(
	HMODULE	hModule,
	DWORD	ul_reason_for_call,
	LPVOID	lpReserved
	)
{
	UNUSED(hModule);
	UNUSED(ul_reason_for_call);
	UNUSED(lpReserved);

	return TRUE;
}


/*
 * prov_time_pps_create - create PPS handle given underlying device
 */
int WINAPI
prov_time_pps_create(
	HANDLE		device,	/* underlying device */
	pps_handle_t *	handle	/* returned handle */
	)
{
	OLD_SERIAL_PPS_STAMPS	old_pps_stamps;
	DWORD			bytes;
	OVERLAPPED		ol;
	serial_unit *		pserunit;
	pps_unit_t *		punit;

	/*
	 * For this ioctl which will never block, we don't want to go
	 * through the overhead of a completion port, so we use an
	 * event handle in the overlapped structure with its 1 bit set.
	 *
	 * From GetQueuedCompletionStatus docs:
	 * Even if you have passed the function a file handle associated 
	 * with a completion port and a valid OVERLAPPED structure, an 
	 * application can prevent completion port notification. This is 
	 * done by specifying a valid event handle for the hEvent member 
	 * of the OVERLAPPED structure, and setting its low-order bit. A 
	 * valid event handle whose low-order bit is set keeps I/O 
	 * completion from being queued to the completion port.
	 */
	ol.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
	ol.hEvent = (HANDLE) ((ULONG_PTR)ol.hEvent | 1);

	if (FALSE == DeviceIoControl(
		device, 
		IOCTL_SERIAL_GET_PPS_STAMPS, 
		NULL, 
		0, 
		&old_pps_stamps, 
		sizeof(old_pps_stamps), 
		&bytes, 
		&ol)
		|| sizeof(old_pps_stamps) != bytes) {

		/* 
		 * If you want to write some dead code this could detect the 
		 * IOCTL being pended, but the driver always has the info
		 * instantly, so ERROR_IO_PENDING isn't a concern.
		 */
		CloseHandle(ol.hEvent);
		fprintf(stderr,
			"time_pps_create: IOCTL_SERIAL_GET_PPS_STAMPS: %u %u\n",
			bytes,
			GetLastError());

		return ENXIO;
	}

	/*
	 * Allocate and initialize serial unit structure.
	 */

	pserunit = malloc(sizeof(*pserunit));
	if (NULL == pserunit)
		return ENOMEM;

	memset(pserunit, 0, sizeof(*pserunit));
	pserunit->device = device;
	pserunit->ol.hEvent = ol.hEvent;
	pserunit->p_ntp_timestamp_from_counter = p_ntp_timestamp_from_counter;

	*handle = (*p_create_pps_handle)(pserunit);
	if (*handle) {
		punit = (pps_unit_t *)*handle;
		punit->params.api_version = PPS_API_VERS_1;
		punit->params.mode = PPS_CAPTUREASSERT | PPS_TSFMT_TSPEC;
	}

	return (*handle)
		   ? 0
		   : ENOMEM;
}


/*
 * prov_time_pps_destroy - release PPS handle
 */
int WINAPI
prov_time_pps_destroy(
	pps_unit_t *	unit,
	void *		context
	)
{
	serial_unit *pserunit;

	UNUSED(unit);

	pserunit = context;
	CloseHandle(pserunit->ol.hEvent);
	free(pserunit);

	return 0;
}


/*
 * prov_time_pps_setparams - set parameters for handle
 */
int WINAPI
prov_time_pps_setparams(
	pps_unit_t *		unit,
	void *			context,
	const pps_params_t *	params
	)
{
	serial_unit *	pserunit;
	int		mode, mode_in;

	pserunit = context;

	/*
	 * There was no reasonable consensus in the API working group.
	 * I require `api_version' to be set!
	 */
	if (params->api_version != PPS_API_VERS_1)
		return EINVAL;

	/*
	 * The only settable modes are PPS_CAPTUREASSERT,
	 * PPS_OFFSETASSERT, and the timestamp formats.
	 */
	mode_in = params->mode;

	/*
	 * Only one of the time formats may be selected
	 * if a nonzero assert offset is supplied.
	 */
	if ((mode_in & (PPS_TSFMT_TSPEC | PPS_TSFMT_NTPFP))
	    == (PPS_TSFMT_TSPEC | PPS_TSFMT_NTPFP)) {

		if (params->assert_offset.tv_sec ||
		    params->assert_offset.tv_nsec) 
			return EINVAL;

		/*
		 * If no offset was specified but both time
		 * format flags are used consider it harmless
		 * but turn off PPS_TSFMT_NTPFP so getparams
		 * will not show both formats lit.
		 */
		mode_in &= ~PPS_TSFMT_NTPFP;
	}

	/* turn off read-only bits */
	mode_in &= ~SERIALPPS_RO;

	/*
	 * test remaining bits, should only have captureassert, 
	 * offsetassert, and/or timestamp format bits.
	 */
	if (mode_in & ~(PPS_CAPTUREASSERT | PPS_OFFSETASSERT |
			PPS_TSFMT_TSPEC | PPS_TSFMT_NTPFP))
		return EOPNOTSUPP;

	/*
	 * ok, ready to go.
	 */
	mode = unit->params.mode;
	unit->params = *params;
	unit->params.mode = mode | mode_in;

	return 0;
}


/*
 * prov_time_pps_fetch - Fetch timestamps
 */

int WINAPI
prov_time_pps_fetch(
	pps_unit_t *		unit,
	void *			context,
	const int		tsformat,
	pps_info_t *		pinfo,
	const struct timespec *	timeout
	)
{
	serial_unit *		pserunit;
	SERIAL_PPS_STAMPS	pps_stamps;
	pps_info_t		infobuf;
	BOOL			rc;
	DWORD			bytes;
	DWORD			lasterr;

	/*
	 * nb. PPS_CANWAIT is NOT set by the implementation, we can totally
	 * ignore the timeout variable.
	 */
	UNUSED(timeout);
	pserunit = context;

	memset(&infobuf, 0, sizeof(infobuf));

	/*
	 * if not captureassert, nothing to return.
	 */
	if (!(unit->params.mode & PPS_CAPTUREASSERT)) {
		*pinfo = infobuf;

		return 0;
	}

	/*
	 * First rev of serialpps.sys didn't support the SeqNum field,
	 * support it by simply returning constant 0 for serial in that case.
	 */
	pps_stamps.SeqNum = 0;

	/*
	 * interrogate (hopefully) serialpps.sys
	 * if it's the standard serial.sys or another driver,
	 * IOCTL_SERIAL_GET_PPS_STAMPS is most likely unknown
	 * and will result in ERROR_INVALID_PARAMETER.
	 */
	bytes = 0;

	rc = DeviceIoControl(
		pserunit->device,
		IOCTL_SERIAL_GET_PPS_STAMPS,
		NULL,
		0,
		&pps_stamps,
		sizeof(pps_stamps),
		&bytes,
		&pserunit->ol);

	if (!rc) {

		lasterr = GetLastError();
		if (ERROR_INVALID_PARAMETER != lasterr) 
			fprintf(stderr, "time_pps_fetch: ioctl err %u\n", 
					lasterr);

		return EOPNOTSUPP;

	} else if (bytes != sizeof(pps_stamps) &&
		   bytes != sizeof(OLD_SERIAL_PPS_STAMPS)) {

		fprintf(stderr, 
			"time_pps_fetch: wanted %u or %u bytes got %u from "
			"IOCTL_SERIAL_GET_PPS_STAMPS 0x%x\n" ,
			sizeof(OLD_SERIAL_PPS_STAMPS),
			sizeof(SERIAL_PPS_STAMPS),
			bytes,
			IOCTL_SERIAL_GET_PPS_STAMPS);

		return ENXIO;
	}

	/*
	 * Interpolating from the high-performance counter will not
	 * yield identical output given identical input over time, as
	 * the baseline timestamp and counter value changes over time.
	 * http://bugs.ntp.org/1965 describes how those small changes
	 * trick ntpd into believing the PPS is active after it is
	 * unplugged from the system.  To ensure repeated fetches
	 * between PPS events return identical timestamps, do not
	 * reconvert from counter to timestamp if we have already for
	 * the same PPS event, instead regurgitate the prior result.
	 */
	if (pps_stamps.SeqNum != 0 &&
	    pps_stamps.SeqNum == pserunit->prior_seq) {
		infobuf.assert_timestamp_ntpfp = pserunit->prior_stamp;
	} else {
		/*
		 * pps_ntp_timestamp_from_counter takes the two flavors
		 * of timestamp we have (counter and system time) and
		 * uses whichever it can to give the best NTP fixed-
		 * point conversion.  In ntpd the Counterstamp is
		 * used if ntpd is interpolating a higher-precision
		 * clock using the performance counter, otherwise the
		 * Windows timestamp is used.  A stub implementation in
		 * timepps.h simply converts from Windows timestamp to
		 * NTP fixed-point.  We call through a pointer to ntpd's
		 * version.
		 */
		(*pserunit->p_ntp_timestamp_from_counter)(
			&infobuf.assert_timestamp_ntpfp, 
			pps_stamps.Timestamp.QuadPart, 
			pps_stamps.Counterstamp.QuadPart);
		pserunit->prior_seq = pps_stamps.SeqNum;
		pserunit->prior_stamp = infobuf.assert_timestamp_ntpfp;
	}

	/*
	 * Note: only assert timestamps are captured by this
	 * implementation.
	 */
	infobuf.assert_sequence = pps_stamps.SeqNum;

	/*
	 * Apply offset and translate to specified format
	 */
	switch (tsformat) {
	case PPS_TSFMT_NTPFP:	/* NTP format requires no translation */
		if (unit->params.mode & PPS_OFFSETASSERT) {
			NTPFP_L_ADDS(&infobuf.assert_timestamp_ntpfp, 
				     &unit->params.assert_offset_ntpfp);
		}
		break;		

	case PPS_TSFMT_TSPEC:	/* timespec format requires conversion to nsecs form */
		PPS_NTPTOTSPEC(infobuf.assert_timestamp);
		if (unit->params.mode & PPS_OFFSETASSERT) {
			infobuf.assert_timestamp.tv_sec  += 
				unit->params.assert_offset.tv_sec;
			infobuf.assert_timestamp.tv_nsec += 
				unit->params.assert_offset.tv_nsec;
			PPS_NORMALIZE(infobuf.assert_timestamp);
		}
		break;

	default:
		return EINVAL;
	}

	infobuf.current_mode = unit->params.mode;
	*pinfo = infobuf;

	return 0;
}


/*
 * prov_time_pps_kcbind - specify kernel consumer
 *
 * Not supported so far by Windows.
 */
int WINAPI
prov_time_pps_kcbind(
	pps_unit_t *	punit,
	void *		context,
	const int	kernel_consumer,
	const int	edge,
	const int	tsformat
	)
{
	UNUSED(punit);
	UNUSED(context);
	UNUSED(kernel_consumer);
	UNUSED(edge);
	UNUSED(tsformat);

	return EOPNOTSUPP;
}


/*
 * prov_init - returns capabilities and provider name
 */
int WINAPI
ppsapi_prov_init(
	int				ppsapi_timepps_prov_ver,
	pcreate_pps_handle		create_pps_handle,
	ppps_ntp_timestamp_from_counter ntp_timestamp_from_counter,
	char *				short_name_buf,
	size_t				short_name_size,
	char *				full_name_buf,
	size_t				full_name_size
	)
{
	ppsapi_provider test_prov;

	if (ppsapi_timepps_prov_ver < PPSAPI_TIMEPPS_PROV_VER)
		return 0;

	p_create_pps_handle = create_pps_handle;
	p_ntp_timestamp_from_counter = ntp_timestamp_from_counter;

	strncpy(short_name_buf, "serialpps", short_name_size);
	strncpy(full_name_buf, 
		"serialpps.sys, serial.sys with CD timestamping added",
		full_name_size);

	/*
	 * Use function pointer prototypes from timepps.h to verify
	 * our prototypes match with some otherwise pointless code.
	 */
	test_prov.ptime_pps_create = &prov_time_pps_create;
	test_prov.ptime_pps_destroy = &prov_time_pps_destroy;
	test_prov.ptime_pps_fetch = &prov_time_pps_fetch;
	test_prov.ptime_pps_kcbind = &prov_time_pps_kcbind;
	test_prov.ptime_pps_setparams = &prov_time_pps_setparams;
	
	return SERIALPPS_CAPS;
}
