// getaddrinfo-test.cpp
//
// Tests getaddrinfo() with AF_UNSPEC, AF_INET, and AF_INET6
// to demonstrate a bug observed in Windows 11 22H2 where
// a system with both IPv4 and IPv6 internet access receives
// only IPv4 addresses with AF_UNSPEC, which is documented
// to return both.  Not returning IPv6 addresses (other
// than link-local) makes sense if the system has no
// global IPv6 addresses available, but that wasn't the
// situation when the bug was observed.
//

#include <iostream>
#include <ws2tcpip.h>

#define COUNTOF(arr)	(sizeof(arr) / sizeof(arr[0]))

typedef struct name_fam_tag {
	const char *	name;
	int		family;
} name_fam;

name_fam iter[] = {
	{ "AF_UNSPEC", AF_UNSPEC },
	{ "AF_INET", AF_INET },
	{ "AF_INET6", AF_INET6 }
};

void	resolve(const char *hostname, const char *service, int family, const char *fam_name);


int main(int argc, char **argv)
{
	WSADATA		wsa_data;
	const char *	host;
	const char *	service;
	int		i;

	WSAStartup(MAKEWORD(2, 2), &wsa_data);

	if (argc > 1) {
		host = argv[1];
	} else {
		host = "2.pool.ntp.org.";
	}

	if (argc > 2) {
		service = argv[2];
	} else {
		service = "ntp";
	}

	std::cout << "Using hostname " << host << " service " << service << "\n";

	for (i = 0; i < COUNTOF(iter); i++) {
		resolve(host, service, iter[i].family, iter[i].name);
	}

	return 0;
}


void	resolve(const char *hostname, const char *service, int family, const char *fam_name)
{
	int			ret;
	addrinfo		hints, *list;
	char			addr_text[86];
	struct sockaddr_in *	psa4;
	struct sockaddr_in6 *	psa6;
	void *			addr;
	const char *		str_ret;

	std::cout << fam_name << ":\n";

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = family;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_protocol = IPPROTO_UDP;

	ret = getaddrinfo(
		hostname,
		service,
		&hints,
		&list
		);

	if (ret != 0) {
		std::cout << "getaddrinfo error: " << gai_strerror(ret) << "\n";
		return;
	}

	while (list != NULL) {
		switch (list->ai_family) {
		case AF_INET:
			psa4 = (struct sockaddr_in *)(list->ai_addr);
			addr = &psa4->sin_addr;
			break;

		case AF_INET6:
			psa6 = (struct sockaddr_in6 *)(list->ai_addr);
			addr = &psa6->sin6_addr;
			break;

		default:
			std::cout << "Unrecognized address family" << list->ai_family << "\n";
			goto next;
		}

		str_ret = inet_ntop(
			list->ai_family,
			addr,
			addr_text,
			sizeof(addr_text)
			);

		if (NULL == str_ret) {
			std::cout << "inet_ntop() error " << WSAGetLastError() << "\n";
			return;
		}

		std::cout << "\t" << addr_text << "\n";

	    next:
		list = list->ai_next;
	}
}

// Run program: Ctrl + F5 or Debug > Start Without Debugging menu
// Debug program: F5 or Debug > Start Debugging menu
