/*
 *	digest support for NTP, MD5 and with OpenSSL more
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ntp_fp.h"
#include "ntp_string.h"
#include "ntp_stdlib.h"
#include "ntp.h"
#include "isc/string.h"

typedef struct {
	const void *	buf;
	size_t		len;
} robuffT;

typedef struct {
	void *		buf;
	size_t		len;
} rwbuffT;


#if defined(OPENSSL) && defined(ENABLE_CMAC)
static size_t
cmac_ctx_size(
	CMAC_CTX *	ctx
	)
{
	size_t mlen = 0;

	if (ctx) {
		EVP_CIPHER_CTX * 	cctx;
		if (NULL != (cctx = CMAC_CTX_get0_cipher_ctx (ctx)))
			mlen = EVP_CIPHER_CTX_block_size(cctx);
	}
	return mlen;
}
#endif	/* OPENSSL && ENABLE_CMAC */


static EVP_MD_CTX *
get_md_ctx(
	int		type
	)
{
#ifndef OPENSSL
	return emalloc(sizeof(MD5_CTX));
#else
	EVP_MD_CTX *	ctx;
#  if OPENSSL_VERSION_NUMBER >= 0x30000000L
	EVP_MD *	md;
#  endif
	const char *	md_name;

	ctx = EVP_MD_CTX_new();
	md_name = OBJ_nid2sn(type);

#  if OPENSSL_VERSION_NUMBER >= 0x30000000L
	/*
	 * See section FIPS Provider:
	 * https://www.openssl.org/docs/man3.0/man7/crypto.html
	 * for property query strings
	 */
	if (NID_md5 == type) {
		/* MD5 is not used as a crypto hash here. */
		md = EVP_MD_fetch(NULL, md_name, "fips=no");
	} else {
		md = EVP_MD_fetch(NULL, md_name, "");
	}
	if (!EVP_DigestInit_ex(ctx, md, NULL)) {
		msyslog(LOG_ERR, "%s init failed", md_name);
		exit(1);
	}
	EVP_MD_free(md);
#  else	/* OPENSSL_VERSION_NUMBER < 0x30000000L follows */
#    ifdef EVP_MD_CTX_FLAG_NON_FIPS_ALLOW
	if (NID_md5 == type) {
		EVP_MD_CTX_set_flags(ctx, EVP_MD_CTX_FLAG_NON_FIPS_ALLOW);
	}
#    endif
	if (!EVP_DigestInit_ex(ctx, EVP_get_digestbynid(type), NULL)) {
		msyslog(LOG_ERR, "%s init failed", md_name);
		exit(1);
	}
#  endif	/* OPENSSL_VERSION_NUMBER */
	return ctx;
#endif	/* OPENSSL */
}


static size_t
make_mac(
	const rwbuffT *	digest,
	int		ktype,
	const robuffT *	key,
	const robuffT *	msg
	)
{
	/*
	 * Compute digest of key concatenated with packet. Note: the
	 * key type and digest type have been verified when the key
	 * was created.
	 */
	size_t	retlen = 0;

#ifdef OPENSSL

	INIT_SSL();

	/* Check if CMAC key type specific code required */
#   ifdef ENABLE_CMAC
	if (ktype == NID_cmac) {
		CMAC_CTX *	ctx    = NULL;
		void const *	keyptr = key->buf;
		u_char		keybuf[AES_128_KEY_SIZE];

		/* adjust key size (zero padded buffer) if necessary */
		if (AES_128_KEY_SIZE > key->len) {
			memcpy(keybuf, keyptr, key->len);
			zero_mem((keybuf + key->len),
				 (AES_128_KEY_SIZE - key->len));
			keyptr = keybuf;
		}

		if (NULL == (ctx = CMAC_CTX_new())) {
			msyslog(LOG_ERR, "MAC encrypt: CMAC %s CTX new failed.", CMAC);
			goto cmac_fail;
		}
		if (!CMAC_Init(ctx, keyptr, AES_128_KEY_SIZE, EVP_aes_128_cbc(), NULL)) {
			msyslog(LOG_ERR, "MAC encrypt: CMAC %s Init failed.",    CMAC);
			goto cmac_fail;
		}
		if (cmac_ctx_size(ctx) > digest->len) {
			msyslog(LOG_ERR, "MAC encrypt: CMAC %s buf too small.",  CMAC);
			goto cmac_fail;
		}
		if (!CMAC_Update(ctx, msg->buf, msg->len)) {
			msyslog(LOG_ERR, "MAC encrypt: CMAC %s Update failed.",  CMAC);
			goto cmac_fail;
		}
		if (!CMAC_Final(ctx, digest->buf, &retlen)) {
			msyslog(LOG_ERR, "MAC encrypt: CMAC %s Final failed.",   CMAC);
			retlen = 0;
		}
	  cmac_fail:
		if (ctx)
			CMAC_CTX_free(ctx);
	}
	else
#   endif /* ENABLE_CMAC */
	{	/* generic MAC handling */
		EVP_MD_CTX *	ctx;
		u_int		uilen = 0;

		ctx = get_md_ctx(ktype);
		if (NULL == ctx) {
			goto mac_fail;
		}
		if ((size_t)EVP_MD_CTX_size(ctx) > digest->len) {
			msyslog(LOG_ERR, "MAC encrypt: MAC %s buf too small.",
				OBJ_nid2sn(ktype));
			goto mac_fail;
		}
		if (!EVP_DigestUpdate(ctx, key->buf, (u_int)key->len)) {
			msyslog(LOG_ERR, "MAC encrypt: MAC %s Digest Update key failed.",
				OBJ_nid2sn(ktype));
			goto mac_fail;
		}
		if (!EVP_DigestUpdate(ctx, msg->buf, (u_int)msg->len)) {
			msyslog(LOG_ERR, "MAC encrypt: MAC %s Digest Update data failed.",
				OBJ_nid2sn(ktype));
			goto mac_fail;
		}
		if (!EVP_DigestFinal(ctx, digest->buf, &uilen)) {
			msyslog(LOG_ERR, "MAC encrypt: MAC %s Digest Final failed.",
				OBJ_nid2sn(ktype));
			uilen = 0;
		}
	  mac_fail:
		retlen = (size_t)uilen;

		if (ctx)
			EVP_MD_CTX_free(ctx);
	}

#else /* !OPENSSL follows */

	if (NID_md5 == ktype) {
		EVP_MD_CTX *	ctx = emalloc(sizeof(MD5_CTX));
		size_t		len = 0;

		if (digest->len < MD5_LENGTH) {
			msyslog(LOG_ERR, "%s", "MAC encrypt: MAC md5 buf too small.");
		} else {
			MD5Init(ctx);
			MD5Update(ctx, (const void *)key->buf, key->len);
			MD5Update(ctx, (const void *)msg->buf, msg->len);
			MD5Final(digest->buf, ctx);
			len = MD5_LENGTH;
		}
		free(ctx);
		retlen = len;
	} else {
		msyslog(LOG_ERR, "MAC encrypt: invalid key type %d", ktype);
	}

#endif /* !OPENSSL */

	return retlen;
}


/*
 * MD5authencrypt - generate message digest
 *
 * Returns length of MAC including key ID and digest.
 */
size_t
MD5authencrypt(
	int		type,	/* hash algorithm */
	const u_char *	key,	/* key pointer */
	size_t		klen,	/* key length */
	u_int32 *	pkt,	/* packet pointer */
	size_t		length	/* packet length */
	)
{
	u_char	digest[EVP_MAX_MD_SIZE];
	rwbuffT digb = { digest, sizeof(digest) };
	robuffT keyb = { key, klen };
	robuffT msgb = { pkt, length };
	size_t	dlen = 0;

	dlen = make_mac(&digb, type, &keyb, &msgb);
	/* If the MAC is longer than the MAX then truncate it. */
	if (dlen > MAX_MDG_LEN)
		dlen = MAX_MDG_LEN;
	memcpy((u_char *)pkt + length + KEY_MAC_LEN, digest, dlen);
	return (dlen + KEY_MAC_LEN);
}


/*
 * MD5authdecrypt - verify MD5 message authenticator
 *
 * Returns one if digest valid, zero if invalid.
 */
int
MD5authdecrypt(
	int		type,	/* hash algorithm */
	const u_char *	key,	/* key pointer */
	size_t		klen,	/* key length */
	u_int32	*	pkt,	/* packet pointer */
	size_t		length,	/* packet length */
	size_t		size,	/* MAC size */
	keyid_t		keyno   /* key id (for err log) */
	)
{
	u_char	digest[EVP_MAX_MD_SIZE];
	rwbuffT digb = { digest, sizeof(digest) };
	robuffT keyb = { key, klen };
	robuffT msgb = { pkt, length };
	size_t	dlen = 0;

	dlen = make_mac(&digb, type, &keyb, &msgb);

	/* If the MAC is longer than the MAX then truncate it. */
	if (dlen > MAX_MDG_LEN)
		dlen = MAX_MDG_LEN;
	if (size != (size_t)dlen + KEY_MAC_LEN) {
		msyslog(LOG_ERR,
			"MAC decrypt: MAC length error: len=%u key=%d",
			(u_int)size, keyno);
		return (0);
	}
	return !isc_tsmemcmp(digest,
		 (u_char *)pkt + length + KEY_MAC_LEN, dlen);
}

/*
 * Calculate the reference id from the address. If it is an IPv4
 * address, use it as is. If it is an IPv6 address, do a md5 on
 * it and use the bottom 4 bytes.
 * The result is in network byte order.
 */
u_int32
addr2refid(sockaddr_u *addr)
{
	u_char		digest[EVP_MAX_MD_SIZE];
	u_int32		addr_refid;
	EVP_MD_CTX *	ctx;
	u_int		len;

	if (IS_IPV4(addr)) {
		return (NSRCADR(addr));
	}
	INIT_SSL();
	ctx = get_md_ctx(NID_md5);
	EVP_DigestUpdate(ctx, (u_char *)&SOCK_ADDR6(addr),
			 sizeof(SOCK_ADDR6(addr)));
	EVP_DigestFinal(ctx, digest, &len);
	EVP_MD_CTX_free(ctx);
	memcpy(&addr_refid, digest, sizeof(addr_refid));
	return (addr_refid);
}
