/*
 * atomic/gcc_x86_x64.h - atomic operations for X86 and X64 (aka amd64)
 *
 * Written by Juergen Perlinger (perlinger@ntp.org) for the NTP project.
 * The contents of 'html/copyright.html' apply.
 *
 * ----------------------------------------------------------------------
 *
 * These versions uses GCC inline assembly
 */
#ifndef ATOMIC_GCC_X86_X64_H
#define ATOMIC_GCC_X86_X64_H

static inline uint32_t atomic_inc(
	volatile uint32_t *dst)
{
	uint32_t v = 1;
	__asm__ volatile("lock xaddl %0,%1"
			 : "+r"(v) : "m"(*dst) : "memory");
	return v + 1;
}

static inline uint32_t atomic_set(
	volatile uint32_t *dst,
	uint32_t           val)
{
	__asm__ volatile("lock xchgl %0,%1"
			 : "+r"(val) : "m"(*dst) : "memory");
	return val;
}

static inline uint32_t atomic_cas(
	volatile uint32_t *dst,
	uint32_t           val,
	uint32_t           cmp)
{
	__asm__ volatile("lock cmpxchgl %1,%2"
			 : "+a"(cmp) : "r"(val), "m"(*dst)  : "memory");
	return cmp;
}

/* full barrier uses a kernel trick: atomically add 0 to the location
 * targeted by the stack pointer. Guaranteed to be a valid address, but
 * the operand name depends on the target...
 */
static inline void atomic_barrier(void)
{
	uint32_t dummy = 0;
	__asm__ volatile(
#  if defined(__x86_64__)
		"lock xaddl %0,(%%rsp)"
#  else
		"lock xaddl %0,(%%esp)"
#  endif
		: "+r"(dummy));
}

#endif /* !defined(ATOMIC_GCC_X86_X64_H) */
