/*
 * libshmclk.h -- API definitions for libshmclk
 *
 * Written by Juergen Perlinger (perlinger@ntp.org) for the NTP project.
 * The contents of 'html/copyright.html' apply.
 *
 * ----------------------------------------------------------------------
 */
#ifndef LIBSHMCLK_H
#define LIBSHMCLK_H

/* Here comes the usual Win32 DLL API dance */
#if !defined(LIBSHMCLK_API)
# if defined(_MSC_VER) && defined(LIBSHMCLK_DLL)
#  if LIBSHMCLK_EXPORTS
#   define LIBSHMCLK_API __declspec(dllexport)
#  else
#   define LIBSHMCLK_API __declspec(dllimport)
#  endif
# else
#  define LIBSHMCLK_API extern
# endif
#endif

#ifdef __cplusplus
extern "C" {
#endif

struct ntp_shmclk_dsc;
typedef struct ntp_shmclk_dsc * ntp_shmclk_handle;

/*! @brief auxiliary data tags */
enum ntp_shmclk_aux {
	ntp_shmclk_aux_none,
	ntp_shmclk_aux_leap,
	ntp_shmclk_aux_pps,
	ntp_shmclk_aux_poll,
	ntp_shmclk_aux_prec
};

/*! @brief fractional time stamp format
 *
 * For those who do not trust (or simply have no good version of) the
 * 'timeval' and 'timespec' structs (Win32 again): Here we have a
 * fractional time stamp format that leaves no room for speculations
 * about component sizes.
 */
struct shmclk_llfp_t {
        int64_t  fp_secs;       /* seconds since 1970-01-01,00:00:00Z*/
        uint32_t fp_frac;       /* binary fractional second (Q0.32)  */
};
typedef struct shmclk_llfp_t shmclk_llfp_t;

/*! @brief open a NTPD shared memory clock
 *
 * Open a shared memory clock unit of NTPD, using the indicated clock
 * unit and operation mode.
 *
 * @param[in] unit the clock unit to use
 * @param[in] the operation mode (currently 0 or 1)
 *
 * @return a handle that must be used for further access, or NULL in
 * case of errors. Inspect errno for further information.
 */
ntp_shmclk_handle LIBSHMCLK_API
ntp_shmclk_open(int unit, int mode);

/*! @brief close a NTPD shared memory clock
 *
 * Close a shared memory clock unit. The handle mst be a value returned
 * by ntp_shmclk_open(), including a possible NULL handle. Release all
 * internal ressources associated with the clock.
 *
 * @param[in] chnd the clock handle to shut down
 * @return    0 on success, -1 on error (check errno for details)
 */
int LIBSHMCLK_API
ntp_shmclk_close(ntp_shmclk_handle chnd);

/*! @brief clock/daemon live check
 *
 * Check if the shared memory clock is healthy and alive. This cannot
 * always immediately decided, and in this case, the check must be
 * repeated somewhat later until a decisive result can be obtained.
 *
 * This also indicates to the daemon the client is still alive and can
 * be called periodically (recommended: 1..5 sec) purely for this side
 * effect.
 *
 * @param[in] chnd the clock handle to check
 * @return    1 - alive
 *            0 - dead
 *           -1 - don't know yet
 */
int LIBSHMCLK_API
ntp_shmclk_alive(ntp_shmclk_handle chnd);

/* @brief set timestamp from struct timeval
 *
 * Provide a time sample in usec precision, together with a leap
 * indicator.
 *
 * @param[in] chnd     the clock handle to use
 * @param[in] reftime  reference time (the external/remote one)
 * @param[in] rcvtime  receive time (the internal/local one)
 * @return 0 if ok
 *        -1 if error (consult errno for details)
 */
int LIBSHMCLK_API
ntp_shmclk_sample_tv(ntp_shmclk_handle      chnd   ,
		     const struct timeval * reftime,
		     const struct timeval * rcvtime);

/* @brief set timestamp from struct timespec
 *
 * Provide a time sample in nsec precision, together with a leap
 * indicator.
 *
 * @param[in] chnd     the clock handle to use
 * @param[in] reftime  reference time (the external/remote one)
 * @param[in] rcvtime  receive time (the internal/local one)
 * @return 0 if ok
 *        -1 if error (consult errno for details)
 */
int LIBSHMCLK_API
ntp_shmclk_sample_ts(ntp_shmclk_handle       chnd   ,
		     const struct timespec * reftime,
		     const struct timespec * rcvtime);

/* @brief set timestamp from fractional times
 *
 * Provide a time sample with binary fractions, together with a leap
 * indicator.
 *
 * @param[in] chnd     the clock handle to use
 * @param[in] reftime  reference time (the external/remote one)
 * @param[in] rcvtime  receive time (the internal/local one)
 * @return 0 if ok
 *        -1 if error (consult errno for details)
 */
int LIBSHMCLK_API
ntp_shmclk_sample_fp(ntp_shmclk_handle     chnd   ,
		     const shmclk_llfp_t * reftime,
		     const shmclk_llfp_t * rcvtime);

/*! @brief set/get auxiliary information
 *
 * Inform the daemon about the clock precision and the recommended poll
 * cycle, if the clock mode supports this.
 *
 * @param[in]  chnd  the clock handle to use
 * @param[in]  what  the item to get/set
 * @param[in]  nval  the value to set
 * @param[out] oval  pointer where to store the old value (NULL is ok)
 * @return 0 if ok
 *        -1 if error (consult errno for details)
 */
int LIBSHMCLK_API
ntp_shmclk_auxinfo(ntp_shmclk_handle   chnd,
		   enum ntp_shmclk_aux what,
		   int                 nval,
		   int *               oval);

#ifdef SYS_WINNT 

/* @brief convert FILETIME delta to LLFP delta
 *
 * Convert a FILETIME delta into a fractional time stamp. This assumes
 * the FILETIME is effectively a signed(!) 64-bit integer, so beware of
 * dragons!
 */
void LIBSHMCLK_API
FileTimeDeltaToShmClkFP(const FILETIME * in, shmclk_llfp_t * out);

/* @brief convert FILETIME absolute time to LLFP time
 *
 * Shifts the start of the time scales to convert from 100nsec since
 * 1601 to fractional seconds since 1970.
 */
void LIBSHMCLK_API
FileTimeStampToShmClkFP(const FILETIME * in, shmclk_llfp_t * out);

#endif

#ifdef __cplusplus
}
#endif

#endif /* !defined(LIBSHMCLK_H) */
