/*
 * shmclockdef.h -- definitions for the SHM clock driver memory layout
 *
 * Written by Juergen Perlinger (perlinger@ntp.org) for the NTP project.
 * The contents of 'html/copyright.html' apply.
 *
 * ----------------------------------------------------------------------
 * Parts taken from 'refclock_shm.c'
 */

#ifndef SHMCLOCKDEF_H
#define SHMCLOCKDEF_H

#if defined(HAVE_STDINT_H)
# include <stdint.h>
#elif defined(_MSC_VER)
# include "stdint_msvc.h"
#else
# include "stdint_ntp.h"
#endif

/*
 * =====================================================================
 *
 * mode 0 (the 'classic' mode) memory segment layout
 *
 * (This is essentally ripped from the 'old' SHM driver)
 * =====================================================================
 */

typedef struct shm0_data {
	int    mode; /* 0 - if valid is set:
		      *       use values,
		      *       clear valid
		      * 1 - if valid is set:
		      *       if count before and after read of values is equal,
		      *         use values
		      *       clear valid
		      */
	volatile int    count;
	time_t		clockTimeStampSec;
	int		clockTimeStampUSec;
	time_t		receiveTimeStampSec;
	int		receiveTimeStampUSec;
	int		leap;
	int		precision;
	int		nsamples;
	volatile int    valid;
	unsigned	clockTimeStampNSec;	/* Unsigned ns timestamps */
	unsigned	receiveTimeStampNSec;	/* Unsigned ns timestamps */
	int		dummy[8];
} shm0_data_t;

/*
 * =====================================================================
 *
 * mode 1 memory segment layout
 *
 * =====================================================================
 *
 * Mode 1 uses a very different memory layout scheme:
 *
 * In the first 128 bytes we have a header, with magic number,
 * read and write counters, ... 
 * In other word, administrative stuff for the management of the SHM
 * segment.
 *
 * Following this is a circular buffer with 8 entries of 128 bytes that
 * is used for reference / receive time pairs, plus leap and precision
 * information and a time format indicator.
 *
 * Every slot in the circular buffers has a guard value in the first
 * DWORD of the slot, so the effective size for user data is limited to
 * 124 bytes or 31 DWORDS.  Every slot is addressed by a sequence number
 * mod 8, and the sequence number is also the value of the guard.  This
 * permits read/write collision and overwrite detection in cooperation
 * with the 32-bit read/write count sequence numbers, provided both
 * sides agree on the read/write sequence. See below.
 *
 * Like all "collision detect and retry" strategies, the one employed
 * here is no safeguard against starvation -- no lock-free stratagem can
 * provide this.  It's also not a universal collision detection
 * mechanism; it would suffer from the ABA problem when a client is able
 * to do 2^32 updates during on read cycle of the ntp daemon.
 *
 * The strategy chosen here has shown itself to be nearly free of
 * collisions for up to several hundred updates per second for a similar
 * single-producer, single-consumer problem and the SHM clock is
 * expected to have a much slower update rate.
 *
 *  Update order
 *  ------------
 *
 * To work properly, read and write must work in opposite direction, and
 * barrier instructions are needed to make the visibility of changes
 * explicit.
 *
 *  - writer procedure -
 *
 * The writer will first read back the write counter and increment it
 * internally, WITHOUT updating the clock structure. Then he has to
 * select the proper slot according to the new sequence number and write
 * the sequence number to the 1st guard field using a release barrier.
 * Then the remainder of the data field can be updated in any order,
 * though in acending address order is preferred. Then the 2nd gurad
 * field is updated, also using a release barrier. Finally the write
 * sequence number is written back to the header with a final release
 * barrier.
 *
 * - reader procedure -
 *
 * The reader must first read back the write sequence number and adjust
 * it's read sequence number according to the write sequence. Then he
 * selects the proper data slot and reads the 2nd guard with an quire
 * barrier, followed by reading the paylod. Finally the 1st guard is
 * read back with an aquire barrier. If the guards do not match the read
 * sequence number, a collision is detected and the record is
 * dicarded. The barriers in conjunction with propr volatile
 * declarations will ensure that the read operations are executed in the
 * proper order.  Finally the reader updates the read sequence number
 * with a release barrier.
 *
 * If release barriers are not available, full barriers do the trick,
 * too. There might be a slight performance impact, but since the update
 * is a slow-rate process this does not hurt very much.
 *
 *  SHM layout
 *  ----------
 *
 *  base + 000h	 SHM haeder structure
 *
 *  base + 080h	 buffer, slot0 
 *  base + 100h	 buffer, slot1
 *  base + 180h	 buffer, slot2
 *  base + 200h	 buffer, slot3
 *  base + 280h	 buffer, slot4
 *  base + 300h	 buffer, slot5
 *  base + 380h	 buffer, slot6
 *  base + 400h	 buffer, slot7

 *  base + 480h	 first byte beyond
 *
 *  DATA SLOT layout
 *  ----------------
 *
 *  base + 00h  1st guard, DWORD
 *  base + 04h  start of payload
 *  base + 7Ch  2nd guard, DWORD
 */

/* The type1 SHM is accessed in DWORDS (32-bit units), and the size
 * definitions reflect this.
 */
#define SHM1_HEAD_SIZE	 32 /* DWORDS */
#define SHM1_SLOT_SIZE	 32 /* DWORDS */
#define SHM1_SLOT_COUNT	 8  /* SLOTS */

#define SHM1_MAGIC	 0x2AA5FDA2 /* crc32("NTP SHM CLOCK MODE1") */

#define NTP_SHM1_VERSION(hi,lo) (((uint16_t)(hi)<<8) + ((uint16_t)(lo) & 0x00FF))
#define SEQ2OFFSET(n) (((n) % SHM1_SLOT_COUNT) * SHM1_SLOT_SIZE)

/*
 * time stamp format definitions.
 * If a time stamp is present, it consists of an integral and a
 * fractional part. The integral part is always a time_t value, that is,
 * seconds since 1970-01-01,00:00:00. The fractional part can be
 * microseconds, nanoseconds or a 32-bit binary fractional.
 */ 
enum TimeStampFormat {
	TSF_none,	/* no timestamp                                */
	TSF_lfp,	/* binary fractional, as in l_fp               */
	TSF_timespec,	/* nanoseconds, as in a struct timespec        */
	TSF_timeval	/* microseconds, as in struct timeval          */
};

/*
 * structure of a data slot entry in the buffer area.
 * 
 * This must not exceed the size of a data slot or truncation will
 * happen when writing/reading the data.
 *
 * This structure should be tightly packed; the current manual layout
 * should do the trick. Use manual padding if necessary.
 */
typedef struct shm1_data {
	/* 0*/	uint16_t rec_size;	/* ecord size in bytes         */
	/* 2*/	uint16_t rec_type;	/* record type, zero for now   */
	/* 4*/	uint32_t tsformat   :2;	/* see enum TimeStampFormat    */
		/* time stamps follows; valid if tsformat != none:     */
	/* 8*/	int64_t  reftime_secs;	/* seconds since 1970, UTC     */
	/*16*/  int64_t  rcvtime_secs;
	/*24*/  uint32_t reftime_frac;  /* fraction as per tsformat    */
	/*28*/  uint32_t rcvtime_frac;
	/* 32 -- 120
	 * add other data here
	 */
} shm1_data_t;
	

/*
 * SHM type 1 header structure
 *
 */
typedef struct shm1_head {
	/* -- Magic field	 
	 * Will be reset to 0 when the server stops this clock. For
	 * proper live check on the client side, whatch this magic and
	 * the read counter -- if the server dies unexpectedly there
	 * might be a dangling segment with a valid magic field, but
	 * obviously the read counter does not change any more.
	 */
	/* 0*/	uint32_t magic;
	
	/* -- Server SHM protocol and capabilities
	 * The server will publish the version and capabilities here.
	 * A client must behave accordingly and refuse to handle things
	 * he does not understand.
	 */
	/* 4*/	uint16_t server_version;
	/* 6*/	uint16_t server_caps;

	/* -- Size of header and data slots in DWORDs, that is, in
	 * multiples of 32bit units, and the number of data slots.
	 */
	/* 8*/  uint16_t head_dwsize;	/* header size in DWORD (32bit) */
	/*10*/  uint16_t data_dwsize;	/* data slot size*/
	/*12*/  uint16_t data_slots;	/* number of data slots         */
	/*14*/  uint16_t _unused_00;	/* manual padding here */
	
	/* -- Timestamp data sequence counters
	 * All counters might have an undefined initial state, and are
	 * always incremented with wrap-around, which comes naturally
	 * for usigned types. The client has to increment the write
	 * counter with every write, so ntpd can detect changes and
	 * fetch another item; ntpd increments the read counter with
	 * every read attempt that follows a write count change
	 * detection.
	 */
	/*16*/	volatile uint32_t seqnum_wr;
	/*20*/	volatile uint32_t seqnum_rd;
	
	/* -- live ticker for server and client */
	/*24*/	volatile uint32_t server_livetick;
	/*28*/	volatile uint32_t client_livetick;

	/*32*/  volatile uint32_t server_ident;
	/*36*/  volatile uint32_t client_ident;
	
		/* leap indicator, with the normal definitions:
		 *  0 -- nothing
		 *  1 -- leap second insert pending
		 *  2 -- leap second removal pending
		 *  3 -- clock out of sync
		 *  * -- anything else is ignored 
		 */
	/*40*/	volatile uint8_t client_leap;
		/* client clock precision
		 * (ignored if >= 0)
		 */
	/*41*/  volatile int8_t  client_prec;
		/* PPS client flag
		 *  0 -- reset PPS PEER flag
		 *  1 -- set   PPS PEER flag
		 *  * -- ignored
		 */
	/*42*/	volatile uint8_t client_pps;
		/* preferred minmum poll rate for client
		 * (ignored if < 0)
		 */
	/*43*/	volatile int8_t  client_minpoll;

	/*44 -- 128
	 * additional info can be placed her, but make sure the data size
	 * does not exceed the allotted header size.
	 */
} shm1_head_t;


/*
 * ---------------------------------------------------------------------
 * SHM system ressources used for access
 *
 * Note that not all implementations (SYSV vs. Windows) need all fields,
 * nor do all types. But it's easier to handle when the relevant stuff
 * is together.
 */
typedef struct shmclk_sys
{
	void *	sys_addr;	/* assigned base address of SHM segment */
	char *  sys_name;	/* name of object (Win32 and Posix only */
	size_t  sys_size;	/* mapped memory size in raw bytes      */

#if defined(SYS_WINNT)

	/* Under Win32, we have a system handle for the file mapping. */
	HANDLE  sys_handle;

#else

	/* UN*Xish systems will either have a file descriptor (for POSIX
	 * shared memory) that we have to close if no longer needed, or
	 * just an identifier for SysV shared memory, which is NOT a managed
	 * ressource but handy to keep anyway.
	 */
	int	sys_shmid;

#endif
} shmclk_sys_t;

#endif /* !defined(SHMCLOCKDEF_H) */
