/*
 * libshmclk.c -- the SHM clock library API functions
 *
 * Written by Juergen Perlinger (perlinger@ntp.org) for the NTP project.
 * The contents of 'html/copyright.html' apply.
 */

#include <config.h>
#include "internal.h"
#include "libshmclk.h"


ntp_shmclk_handle ntp_shmclk_open(
	int unit,
	int mode)
{
	const struct shmHandlers * handlers;
	struct ntp_shmclk_dsc    * chnd;
	
	switch (mode) {
	case 0:
		handlers = &lsmc_t0_handlers;
		break;
	case 1:
		handlers = &lsmc_t1_handlers;
		break;
	default:
		handlers = NULL;
		break;
	}
	if (!handlers) {
		errno = EINVAL;
		return NULL;
	}
	chnd = calloc(1, sizeof(struct ntp_shmclk_dsc));
	if (!chnd)
		return NULL;

	if ((*handlers->attach)(&chnd->shmdesc, unit)) {		
		chnd->handlers = handlers;
	} else {
		free(chnd);
		chnd = NULL;
	}
	return chnd;
}

int ntp_shmclk_close(
	ntp_shmclk_handle chnd)
{
	if (chnd) {
		(*chnd->handlers->detach)(&chnd->shmdesc);
		free(chnd);
	}
	return 0;
}

int ntp_shmclk_alive(
	ntp_shmclk_handle chnd)
{
	if (!chnd) {
		errno = EINVAL;
		return 0;
	}
	return (*chnd->handlers->alive)(&chnd->shmdesc);
	
}

int ntp_shmclk_sample_tv(
	ntp_shmclk_handle      chnd   ,
	const struct timeval * reftime,
	const struct timeval * rcvtime)
{
	if (!chnd) {
		errno = EINVAL;
		return 1;
	}
	return (*chnd->handlers->pushttv)(&chnd->shmdesc, reftime, rcvtime);
}

int ntp_shmclk_sample_ts(
	ntp_shmclk_handle       chnd   ,
	const struct timespec * reftime,
	const struct timespec * rcvtime)
{
	if (!chnd) {
		errno = EINVAL;
		return 1;
	}
	return (*chnd->handlers->pushtts)(&chnd->shmdesc, reftime, rcvtime);
}

int ntp_shmclk_sample_fp(
	ntp_shmclk_handle            chnd   ,
	const struct shmclk_llfp_t * reftime,
	const struct shmclk_llfp_t * rcvtime)
{
	if (!chnd) {
		errno = EINVAL;
		return 1;
	}
	return (*chnd->handlers->pushtfp)(&chnd->shmdesc, reftime, rcvtime);
}

int ntp_shmclk_auxinfo(
    ntp_shmclk_handle   chnd,
    enum ntp_shmclk_aux what,
    int                 nval,
    int *               oval)
{
	int dummy;
	
	if (!chnd) {
		errno = EINVAL;
		return 1;
	}
	if (!oval)
		oval = &dummy;
	return (*chnd->handlers->pushaux)(&chnd->shmdesc, what, nval, oval);
}

#ifdef SYS_WINNT 

void FileTimeDeltaToShmClkFP(
        const FILETIME * in,
        shmclk_llfp_t * out)
{
        union {
                FILETIME ft;
                int64_t  li;
                uint64_t lu;
        } ft64;
        int64_t q;
        int32_t r;

        /* do a floor division by 10^7 to get seconds and fraction */
        ft64.ft = *in;
        q = ft64.li / INT32_C(10000000);
        r = (int32_t)(ft64.li - q * INT32_C(10000000));
        if (r < 0) {
                q -= 1;
                r += INT32_C(10000000);
        }
        out->fp_secs = q;

        /* scale remaining ticks in second to binary fraction, using
         * effectively a MULHI approach
         */
        ft64.lu = UINT64_C(1844674407371); /* 2^64 / 10^7, rounded) */
        ft64.lu *= r;
        out->fp_frac = ft64.ft.dwHighDateTime
                     += ((int32_t)ft64.ft.dwLowDateTime < 0);
        out->fp_secs += (out->fp_frac < ft64.ft.dwHighDateTime);
}

void FileTimeStampToShmClkFP(
        const FILETIME * in,
        shmclk_llfp_t * out)
{
        static const int64_t ft1970 = INT64_C(11644473600) * 10000000;

        union {
                FILETIME ft;
                int64_t  li;
        } ft64;

        ft64.ft = *in;
        ft64.li -= ft1970;
        FileTimeDeltaToShmClkFP(&ft64.ft, out);
}

#endif
/* -*-EOF-*- */
