/*
 * shmops.c -- raw shared memory operations for SHM clock clients
 *
 * Written by Juergen Perlinger (perlinger@ntp.org) for the NTP project.
 * The contents of 'html/copyright.html' apply.
 *
 * ----------------------------------------------------------------------
 * Attach & detach from a shared memory segment. On Win32 there is only
 * one way to do this. Otherwise, we use POSIX shared memory if it's
 * available, and SysV shared memory otherwise. This is the same
 * decision chain as used in the NTP daemon, so the library and the
 * daemon should be able to rendezvous.
 */

#include <config.h>
#include "internal.h"
#include "ntp_atomic.h"

static const char * logmsg = "SHM clock (unit=%u, mode=%u): %s: %s\n";

/*
 * =====================================================================
 *  helpers
 * =====================================================================
 */

#if defined(SYS_WINNT) || defined(HAVE_SHM_OPEN)

static char *
my_aprintf(
	const char *fmt,
	...            )
{
	va_list va;
	size_t blen, flen;
	char  *buff;

	blen = strlen(fmt) & (SIZE_MAX - 0x7f);
	buff = NULL;
	do {
		free(buff);
		va_start(va, fmt);
		blen += 128;
		buff  = malloc(blen);
		if (buff == NULL)
			exit(1);
		flen  = vsnprintf(buff, blen, fmt, va);
		va_end(va);
	} while (flen >= blen);
	return realloc(buff, flen+1);
}

#endif


/* ================================================================== */
#ifdef SYS_WINNT
/* =====================================================================
 *
 * Win32 shared memory (pagefile backed)
 *
 * =====================================================================
 */

static const char*
_fmt_error(
        DWORD errc)
{
        static char mbuf[128];
        DWORD rc;

        rc = FormatMessageA(
                FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
                NULL, errc, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                mbuf, sizeof(mbuf)-1, NULL);
        if (rc >= sizeof(mbuf)) {
                rc = sizeof(mbuf) - 1;
                mbuf[rc] = '\0';
        }
        while (rc && mbuf[rc-1] <= ' ')
                mbuf[--rc] = '\0';

        return mbuf;
}

size_t
lsmc_getShmTime(
	shmclk_sys_t * dsc ,
	int            mode,
	int            unit)
{
        static const char * names[2] = {
		"Global\\NTP%d" , "Global\\NTP%d.%d" };

        MEMORY_BASIC_INFORMATION minfo;

	if (dsc->sys_addr && dsc->sys_size) {
		errno = EEXIST;
		return 0;
	}

	dsc->sys_name   = my_aprintf(names[mode > 0], unit, mode);	
	dsc->sys_handle = OpenFileMapping(
                                PAGE_READWRITE, FALSE, dsc->sys_name);
	if (dsc->sys_handle == NULL) { /*error*/
                fprintf(stderr, logmsg, unit, mode, "OpenFileMapping",
                        _fmt_error(GetLastError()));
                goto fail;
	}

        /* we map the whole thing into memory and try to find about the size later. */
	dsc->sys_addr = MapViewOfFile(
                                dsc->sys_handle, FILE_MAP_WRITE, 0, 0, 0);
	if (dsc->sys_addr == NULL) { /*error*/
                fprintf(stderr, logmsg, unit, mode, "MapViewOfFile",
                        _fmt_error(GetLastError()));
                goto fail;
	}

        /* now get the size of the mapping object */
        if (VirtualQuery(dsc->sys_addr, &minfo, sizeof(minfo)) == 0) {
                fprintf(stderr, logmsg, unit, mode, "VirtualQuery",
                        _fmt_error(GetLastError()));
                goto fail;
        }
	if (minfo.RegionSize < 16) {
		fprintf(stderr, logmsg, unit, mode,
			"VirtualQuery", "no data");
		goto fail;
	}

	dsc->sys_size = minfo.RegionSize;
	errno = 0;
        return minfo.RegionSize;

fail:
        lsmc_delShmTime(dsc);
        return FALSE;
}

void
lsmc_delShmTime(
	shmclk_sys_t * dsc)
{
	if (dsc->sys_addr)
		UnmapViewOfFile(dsc->sys_addr);
        if (dsc->sys_handle)
	        CloseHandle(dsc->sys_handle);
        free(dsc->sys_name);
	memset(dsc, 0, sizeof(shmclk_sys_t));
}

/* ================================================================== */
#elif HAVE_SYS_MMAN_H && HAVE_SHM_OPEN
/* =====================================================================
 *
 * POSIX compatible SHM access
 *
 * =====================================================================
 */

size_t
lsmc_getShmTime (
	shmclk_sys_t * const dsc ,
	int                  mode,
	int                  unit)
{
	struct stat statbuf;
	
	/* If already attached, bail out without touching anything */
	if (dsc->sys_addr && dsc->sys_size) {
		errno = EEXIST;
		return dsc->sys_size;
	}

	/* create or open the memory segment */
	dsc->sys_name  = my_aprintf("/ntpd-shmclk.%u.%u", unit, mode);
	dsc->sys_shmid = shm_open(dsc->sys_name, O_RDWR, 0/*S_IRWXU*/);
	if (dsc->sys_shmid < 0) {
		fprintf(stderr, logmsg, unit, mode,
			"shm_open", strerror(errno));
		goto fail;
	}
	
	/* get the size of the SHM segment */
	if (fstat(dsc->sys_shmid, &statbuf)) {
		fprintf(stderr, logmsg, unit, mode,
			"fstat", strerror(errno));
		goto fail;
	}
	if (statbuf.st_size < 16) {
		fprintf(stderr, logmsg, unit, mode,
			"fstat", "no data");
		goto fail;
	}
		
	/* map to local address space */
	dsc->sys_addr = mmap(NULL, statbuf.st_size, PROT_READ|PROT_WRITE,
			       MAP_SHARED, dsc->sys_shmid, 0);
	if (dsc->sys_addr == MAP_FAILED) {
		fprintf(stderr, logmsg, unit, mode, "mmap");
		goto fail;
	}
	dsc->sys_size = statbuf.st_size;
	errno = 0;
	return statbuf.st_size;

  fail:
	if (dsc->sys_shmid >= 0)
		close(dsc->sys_shmid);
	if (dsc->sys_name)
		shm_unlink(dsc->sys_name);
	free (dsc->sys_name);

	memset(dsc, 0, sizeof(shmclk_sys_t));
	return 0;
}

void
lsmc_delShmTime(
	shmclk_sys_t * const dsc)
{
	if (dsc->sys_addr && dsc->sys_size) {
		munmap(dsc->sys_addr, dsc->sys_size);
		close(dsc->sys_shmid);
		free (dsc->sys_name);
	}
	memset(dsc, 0, sizeof(shmclk_sys_t));
}

/* ================================================================== */
#else
/* =====================================================================
 *
 * SYSV compatible SHM access
 *
 * =====================================================================
 */

size_t
lsmc_getShmTime (
	shmclk_sys_t * const dsc ,
	int                  mode,
	int                  unit)
{
	int             sid;
	struct shmid_ds sms;
	
	/* If already attached, bail out without touching anything */
	if (dsc->sys_addr && dsc->sys_size) {
		errno = EEXIST;
		return dsc->sys_size;
	}
	
	/* 0x4e545030 is NTP0.
	 * Big units will give non-ascii but that's OK
	 * as long as everybody does it the same way. 
	 */
	sid = 0x4e545030 + mode*256 + unit;

	/* get the system identifier for that segment */
	dsc->sys_shmid = shmget(sid, 0, 0); 
	if (dsc->sys_shmid == -1) { /*error */
		fprintf(stderr, logmsg,	mode, unit,
			"shmget", strerror(errno));
		goto fail;
	}

	/* get the size of the SHM segment */
	if (-1 == shmctl(dsc->sys_shmid, IPC_STAT, &sms)) {
		fprintf(stderr, logmsg,	mode, unit,
			"shmctl[IPC_STAT]", strerror(errno));
		goto fail;
	}
	if (sms.shm_segsz < 16) {
		fprintf(stderr, logmsg, unit, mode,
			"shmctl[IPC_STAT]", "no data");
		goto fail;
	}

	/* map to local address space */
	dsc->sys_addr = shmat(dsc->sys_shmid, NULL, 0);
	if (dsc->sys_addr == (void*)-1) { /* error */
		fprintf(stderr, logmsg,	mode, unit,
			"shmat", strerror(errno));
		goto fail;
	}
	dsc->sys_size = sms.shm_segsz;
	errno = 0;
	return sms.shm_segsz;

  fail:
	memset(dsc, 0, sizeof(shmclk_sys_t));
	return 0;
}

void
lsmc_delShmTime(
	shmclk_sys_t * const dsc)
{
	if (dsc->sys_addr && dsc->sys_size)
		(void)shmdt((char*)dsc->sys_addr);
	memset(dsc, 0, sizeof(shmclk_sys_t));
}

#endif
