/*
 * typ0func.c -- type/mode 0 SHM clock support functions
 *
 * Written by Juergen Perlinger (perlinger@ntp.org) for the NTP project.
 * The contents of 'html/copyright.html' apply.
 *
 * ----------------------------------------------------------------------
 */

#include <config.h>
#include "internal.h"
#include "ntp_atomic.h"
#include "libshmclk.h"

/* ---------------------------------------------------------------------
 * type 0 attach
 */
static int t0_attach(
	ntp_shmdata_dsc_t chp ,
	int               unit)
{
	size_t        memsize;
	shm0_data_t * smem;

	memsize = lsmc_getShmTime(&chp->dsc, 0, unit);
	if (memsize < sizeof(shm0_data_t))
		goto fail;
	
	smem = (shm0_data_t *)chp->dsc.sys_addr;	
	smem->mode = 1;
	atomic_barrier();
	
	return 1;

  fail:
	lsmc_delShmTime(&chp->dsc);
	return 0;
}

/* ---------------------------------------------------------------------
 * type 0 detach
 */
static void t0_detach (
	ntp_shmdata_dsc_t chp)
{
	lsmc_delShmTime(&chp->dsc);
}

/* ---------------------------------------------------------------------
 * type 0 time data push
 */
static int t0_pushttv(
	ntp_shmdata_dsc_t      chp    ,
	const struct timeval  *reftime,
	const struct timeval  *rcvtime)
{
	shm0_data_t * smem;

	smem = (shm0_data_t *)chp->dsc.sys_addr;
	if (smem == NULL) {
		errno = EINVAL;
		return -1;
	}

	if (sizeof(smem->count) == sizeof(uint32_t)) {
		atomic_inc((volatile uint32_t*)&smem->count);
	} else {
		++smem->count;
		atomic_barrier();
	}

	smem->clockTimeStampSec    = reftime->tv_sec;
	smem->clockTimeStampUSec   = reftime->tv_usec;
	smem->clockTimeStampNSec   = 0;
	
	smem->receiveTimeStampSec  = rcvtime->tv_sec;
	smem->receiveTimeStampUSec = rcvtime->tv_usec;
	smem->receiveTimeStampNSec = 0;

	smem->valid = 1;
	atomic_barrier();

	return 0;
}

static int t0_pushtts(
	ntp_shmdata_dsc_t      chp    ,
	const struct timespec *reftime,
	const struct timespec *rcvtime)
{
	shm0_data_t * smem;

	smem = (shm0_data_t *)chp->dsc.sys_addr;
	if (smem == NULL) {
		errno = EINVAL;
		return -1;
	}

	if (sizeof(smem->count) == sizeof(uint32_t)) {
		atomic_inc((volatile uint32_t*)&smem->count);
	} else {
		++smem->count;
		atomic_barrier();
	}
	
	smem->clockTimeStampSec    = reftime->tv_sec;
	smem->clockTimeStampUSec   = reftime->tv_nsec / 1000;
	smem->clockTimeStampNSec   = reftime->tv_nsec;
	
	smem->receiveTimeStampSec  = rcvtime->tv_sec;
	smem->receiveTimeStampUSec = rcvtime->tv_nsec / 1000;
	smem->receiveTimeStampNSec = rcvtime->tv_nsec;

	smem->valid = 1;
	atomic_barrier();

	return 0;
}

static int t0_pushtfp(
	ntp_shmdata_dsc_t           chp    ,
	const struct shmclk_llfp_t *reftime,
	const struct shmclk_llfp_t *rcvtime)
{
        (void)chp;
        (void)reftime;
        (void)rcvtime;

        errno = ENOENT;
        return -1;
}

/* ---------------------------------------------------------------------
 * type 0 auxiliary information (poll cycle passing not supported)
 */
static int t0_pushaux(
	ntp_shmdata_dsc_t chp ,
	int               what,
	int               nval,
	int *             oval)
{
	shm0_data_t * smem;
	int           rc;
	
	smem = (shm0_data_t *)chp->dsc.sys_addr;
	if (smem == NULL) {
		errno = EINVAL;
		return -1;
	}

	switch (what) {
	case ntp_shmclk_aux_none:
		rc = 0;
		break;

	case ntp_shmclk_aux_leap:
		*oval = smem->leap;
		if (nval >= 0 && nval <= 3) {
			smem->leap = nval;
			rc = 0;
		} else {
			errno = ERANGE;
			rc = -1;
		}
		break;

	case ntp_shmclk_aux_prec:
		*oval = smem->precision;
		if (nval < 0) {
			smem->precision = nval;
			rc = 0;
		} else {
			errno = ERANGE;
			rc = -1;
		}
		break;

	default:
		errno = EINVAL;
		rc = -1;
		break;
	}

	return rc;
}

/* ---------------------------------------------------------------------
 * type 0 live check
 */
static int t0_alive(
	ntp_shmdata_dsc_t chp)
{
	return chp->dsc.sys_addr != NULL;
}

/* ---------------------------------------------------------------------
 * handler table
 */
const struct shmHandlers lsmc_t0_handlers = {
	t0_attach,
	t0_detach,
	t0_pushttv,
	t0_pushtts,
        t0_pushtfp,
	t0_pushaux,
	t0_alive
};

/* -*-EOF-*- */
