/*
 * typ1func.c -- type/mode 1 SHM clock support functions
 *
 * Written by Juergen Perlinger (perlinger@ntp.org) for the NTP project.
 * The contents of 'html/copyright.html' apply.
 *
 * ----------------------------------------------------------------------
 */

#include <config.h>
#include "internal.h"
#include "libshmclk.h"
#include "ntp_atomic.h"

/* ---------------------------------------------------------------------
 * we need a simple monotonic(!) clock for timeout checks. This is
 * vastly different for UN*X and Win32, but we can cope.
 */
#ifndef SYS_WINNT

static uint32_t
sys_tick_value(void)
{
	struct tms dummy;
	return (uint32_t)times(&dummy);
}

static uint32_t
sys_tick_rate(void)
{
	static uint32_t tps;
    
	if (!tps)
		tps = (uint32_t)sysconf(_SC_CLK_TCK);
	return tps;
}

#else

static uint32_t __inline
sys_tick_value(void)
{
	return GetTickCount();
}

static uint32_t __inline
sys_tick_rate(void)
{
	return 1000;
}

#endif


/* ---------------------------------------------------------------------
 * get the type 1 head pointer
 */
static volatile shm1_head_t*
get_shm1_head_ptr(
	shmclk_sys_t *desc)
{
	return (shm1_head_t *)desc->sys_addr;
}

/* ---------------------------------------------------------------------
 * get a type 1 data slot pointer
 */
static volatile uint32_t*
get_shm1_data_ptr(
	shmclk_sys_t *desc,
	uint32_t      seqn)
{
	volatile uint32_t    *data;
	volatile shm1_head_t *head;
	uint32_t              snum, slen;
	
	head = (volatile shm1_head_t *)desc->sys_addr;
	if (head == NULL)
		return NULL;
	
	data = (volatile uint32_t*)head
	     + head->head_dwsize;
	snum = head->data_slots;
	slen = head->data_dwsize;
	if (!(snum & (snum-1)))
		data += (seqn & (snum-1)) * slen;
	else
		data += (seqn % snum) * slen;

	return data;
}

/* ---------------------------------------------------------------------
 * push a data record for a type 1 clock
 */
static int
push_shm1_data(
	ntp_shmdata_dsc_t   chp ,
	shm1_data_t        *data)
{
	volatile uint32_t    *dwp;
	volatile shm1_head_t *head;

	uint32_t seq;

	if (chp->dsc.sys_addr == NULL)
		return 0;
	
	head = get_shm1_head_ptr(&chp->dsc);
	seq  = head->seqnum_wr + 1;
	dwp  = get_shm1_data_ptr(&chp->dsc, seq);
	atomic_set(dwp, seq);
	*(volatile shm1_data_t*)(dwp+1) = *data;
	atomic_set(dwp+SHM1_SLOT_SIZE-1, seq);
	atomic_set(&head->seqnum_wr, seq);

	return 1;
}

/* ---------------------------------------------------------------------
 * type 1 attach
 */
static int t1_attach(
	ntp_shmdata_dsc_t chp ,
	int               unit)
{
	size_t memsize;
	size_t totsize;
	volatile shm1_head_t * hdr;
	
	memsize = lsmc_getShmTime(&chp->dsc, 1, unit);
	if (memsize < 32)
		goto fail;
	hdr = get_shm1_head_ptr(&chp->dsc);
	totsize = sizeof(uint32_t) *
	    (hdr->head_dwsize + hdr->data_dwsize * hdr->data_slots);
	if (memsize < totsize)
		goto fail;
	
	if (  hdr->magic          != SHM1_MAGIC
	   || hdr->server_version != NTP_SHM1_VERSION(1,0))
		goto fail;

	atomic_set(&hdr->client_ident, getpid());
	
	return 1;

  fail:
	lsmc_delShmTime(&chp->dsc);
	return 0;
}

/* ---------------------------------------------------------------------
 * type 1 detach
 */
static void t1_detach (
	ntp_shmdata_dsc_t chp)
{
	volatile shm1_head_t * head;
	if (chp->dsc.sys_addr == NULL)
		return;
	
	head = get_shm1_head_ptr(&chp->dsc);
	atomic_cas(&head->client_ident, 0, getpid());
	lsmc_delShmTime(&chp->dsc);
}

/* ---------------------------------------------------------------------
 * type 1 time data push
 */
static int t1_pushttv(
	ntp_shmdata_dsc_t      chp    ,
	const struct timeval  *reftime,
	const struct timeval  *rcvtime)
{
	shm1_data_t cdata;
	
	memset(&cdata, 0, sizeof(shm1_data_t));
	cdata.rec_size = sizeof(cdata);

	cdata.reftime_secs = reftime->tv_sec;
	cdata.reftime_frac = reftime->tv_usec;
	cdata.rcvtime_secs = rcvtime->tv_sec;
	cdata.rcvtime_frac = rcvtime->tv_usec;
	cdata.tsformat     = TSF_timeval;
	return push_shm1_data(chp, &cdata);
}

static int t1_pushtts(
	ntp_shmdata_dsc_t      chp    ,
	const struct timespec *reftime,
	const struct timespec *rcvtime)
{
	shm1_data_t cdata;
	
	memset(&cdata, 0, sizeof(shm1_data_t));
	cdata.rec_size = sizeof(cdata);

	cdata.reftime_secs = reftime->tv_sec;
	cdata.reftime_frac = reftime->tv_nsec;
	cdata.rcvtime_secs = rcvtime->tv_sec;
	cdata.rcvtime_frac = rcvtime->tv_nsec;
	cdata.tsformat     = TSF_timespec;
	return push_shm1_data(chp, &cdata);
}

static int t1_pushtfp(
	ntp_shmdata_dsc_t           chp    ,
	const struct shmclk_llfp_t *reftime,
	const struct shmclk_llfp_t *rcvtime)
{
	shm1_data_t cdata;
	
	memset(&cdata, 0, sizeof(shm1_data_t));
	cdata.rec_size = sizeof(cdata);

	cdata.reftime_secs = reftime->fp_secs;
	cdata.reftime_frac = reftime->fp_frac;
	cdata.rcvtime_secs = rcvtime->fp_secs;
	cdata.rcvtime_frac = rcvtime->fp_frac;
	cdata.tsformat     = TSF_lfp;
	return push_shm1_data(chp, &cdata);
}

/* ---------------------------------------------------------------------
 * type 1 auxiliary data
 */
static int t1_pushaux(
	ntp_shmdata_dsc_t chp ,
	int               what,
	int               nval,
	int *             oval)
{
	volatile shm1_head_t * smem;
	int                    rc;
    
	smem = get_shm1_head_ptr(&chp->dsc);
	
	if (smem == NULL) {
		errno = EINVAL;
		return -1;
	}

	switch (what) {
	case ntp_shmclk_aux_none:
		rc = 0;
		break;

	case ntp_shmclk_aux_leap:
		*oval = smem->client_leap;
		if (nval >= 0 && nval <= 3) {
			smem->client_leap = nval;
			rc = 0;
		} else {
			errno = ERANGE;
			rc = -1;
		}
		break;

	case ntp_shmclk_aux_pps:
		*oval = smem->client_pps;
		if (nval == 0 || nval == 1) {
			smem->client_pps = nval;
			rc = 0;
		} else {
			errno = ERANGE;
			rc = -1;
		}
		break;
		
	case ntp_shmclk_aux_poll:
		*oval = smem->client_minpoll;
		if (nval > 0) {
			smem->client_minpoll = nval;
			rc = 0;
		} else {
			errno = ERANGE;
			rc = -1;
		}
		break;

	case ntp_shmclk_aux_prec:
		*oval = smem->client_prec;
		if (nval < 0) {
			smem->client_prec = nval;
			rc = 0;
		} else {
			errno = ERANGE;
			rc = -1;
		}
		break;

	default:
		errno = EINVAL;
		rc = -1;
		break;
	}

	return rc;
}

/* ---------------------------------------------------------------------
 * type 1 live check
 */
static int t1_alive(
	ntp_shmdata_dsc_t chp)
{
	volatile shm1_head_t *head;
	uint32_t              seqno, ticks, limit;
	
	if (!chp->dsc.sys_addr)
		return 0;

	head = get_shm1_head_ptr(&chp->dsc);
	if (head->magic != SHM1_MAGIC)
		return 0;
	if (head->client_ident != getpid())
		return 0;

	/* since we're still he client, update our own live signal */
	atomic_inc(&head->client_livetick);

	/* now check the server live tick changes, and claim failure if
	 * the last update was too long ago. (current limit is 5 secs;
	 * since the daemon should check every second, this has to be
	 * enough.)
	 */
	seqno = head->server_livetick;
	ticks = sys_tick_value();
	limit = sys_tick_rate() * 5;
	if (chp->live_tick != seqno) {
		/* change detected - update ticks and flag success */
		chp->live_tick = seqno;
		chp->live_time = ticks;
		return 1;
	}
	if (ticks - chp->live_time > limit) {
		/* Live signal timeout -- this clock seems orphaned.
		 * Drag live signal time in safe distance behind, but
		 * update it anyway: We might run into overflow rouble
		 * if we don't because of the limited range of ticks.
		 */
		chp->live_time = ticks - 2*limit;
		return 0;
	}

	/* No clear answer yet - have to retry later. */
	return -1;
}


/* ---------------------------------------------------------------------
 * dispatch table
 */ 
const struct shmHandlers lsmc_t1_handlers = {
	t1_attach,
	t1_detach,
	t1_pushttv,
	t1_pushtts,
	t1_pushtfp,
	t1_pushaux,
	t1_alive
};

/* -*-EOF-*- */
