/*
 * refclock_shm - clock driver for utc via shared memory 
 * - under construction -
 * To add new modes: Extend or union the shmTime-struct. Do not
 * extend/shrink size, because otherwise existing implementations
 * will specify wrong size of shared memory-segment
 * PB 18.3.97
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#ifdef HAVE_SYS_MMAN_H
# include <sys/mman.h>
# include <sys/fcntl.h>
# include <sys/stat.h>
#endif

#if defined(REFCLOCK) && defined(CLOCK_SHM)

#include "ntpd.h"
#undef fileno	
#include "ntp_io.h"
#undef fileno	
#include "ntp_refclock.h"
#undef fileno	
#include "ntp_unixtime.h"
#undef fileno	
#include "ntp_stdlib.h"

#include <ntp_types.h>
#include <ntp_fp.h>

#undef fileno	
#include <ctype.h>
#undef fileno	

#ifndef SYS_WINNT
# include <sys/ipc.h>
# include <sys/shm.h>
# include <assert.h>
# include <unistd.h>
# include <stdio.h>
#endif

#include "timespecops.h"
#include "timevalops.h"
#include "ntp_calendar.h"
#include "shmclockdef.h"
#include "ntp_atomic.h"

/*
 * This driver supports a reference clock attached thru shared memory
 */ 

/*
 * SHM interface definitions
 */
#define PRECISION	(-1)	/* precision assumed (0.5 s) */
#define REFID		"SHM"	/* reference ID */
#define DESCRIPTION	"SHM/Shared memory interface"

#define NSAMPLES	3	/* stages of median filter */
#define SHMMODES	2

	
typedef struct refclockproc refclockproc_t;
typedef struct peer         peer_t;
/*
 * Function prototypes
 */
static int   shm_start	 (int unit, peer_t * const peer);
static void  shm_shutdown(int unit, peer_t * const peer);
static void  shm_poll	 (int unit, peer_t * const peer);
static void  shm_timer	 (int unit, peer_t * const peer);
static void  shm_control (int, const struct refclockstat * const,
			  struct refclockstat * const out_st,
			  peer_t * const peer);

static void  shm_clockstats(int, peer_t * const);
static char *my_aprintf(const char *fmt, ...);

/* ----------------------------------------------------------------------
 * Transfer vector
 */
struct	refclock refclock_shm = {
	shm_start,		/* start up driver */
	shm_shutdown,		/* shut down driver */
	shm_poll,		/* transmit poll message */
	shm_control,		/* control (config changes) */
	noentry,		/* not used: init */
	noentry,		/* not used: buginfo */
	shm_timer,		/* once per second */
};

/* ----------------------------------------------------------------------
 * clock unit (we need some forward declarations here)
 */
struct dispatch;
typedef const struct dispatch * dispatchTable;

typedef struct shmunit {
	/* administrative stuff: Make this more like an object, and
	 * provide the necessary back links.
	 */
	dispatchTable	 	handlers;	/* poor man's VFT */
	int			unit;
	peer_t *		peer;
	refclockproc_t *	proc;

	shmclk_sys_t 		smem;
	
	/* debugging/monitoring counters - reset when printed */
	int ticks;		/* number of attempts to read data*/
	int good;		/* number of valid samples */
	int notready;		/* number of peeks without data ready */
	int bad;		/* number of invalid samples */
	int clash;		/* number of access clashes while reading */

	s_char	prec;
	u_char	leap;
	u_char  min_poll;
	u_char  with_pps;

	/* type1 specific data */
	u_short active;
} shmunit_t;


struct dispatch {
	int  (*attach )(shmunit_t * const);
	void (*detach )(shmunit_t * const);
	void (*process)(shmunit_t * const);
};

static int  attachTyp0 (shmunit_t * const);
static void detachTyp0 (shmunit_t * const);
static void processTyp0(shmunit_t * const);

static int  attachTyp1 (shmunit_t * const);
static void detachTyp1 (shmunit_t * const);
static void processTyp1(shmunit_t * const);

static const struct dispatch ShmHandlers[SHMMODES]= {
	{
		attachTyp0, detachTyp0, processTyp0
	},{
		attachTyp1, detachTyp1, processTyp1
	}
};


/* structures and other data types */

static volatile shm1_head_t*
get_shm1_head_ptr(
	shmclk_sys_t *desc)
{
	return (shm1_head_t *)desc->sys_addr;
}

static volatile u_int32*
get_shm1_data_ptr(
	shmclk_sys_t *desc,
	u_int32       slot)
{
	u_int32 *res;
	res = (u_int32 *)desc->sys_addr;
	if (res)
		res += SHM1_HEAD_SIZE
		     + SHM1_SLOT_SIZE * (slot % SHM1_SLOT_COUNT);
	return res;
}

static long poffs(
	const shmclk_sys_t *desc,
	volatile const void *anyp)
{
	return (volatile const char*)anyp - (volatile const char*)desc->sys_addr;
}

/*
 * readDataSLot(...) -- read a data slot with collision check
 *
 * Read the data slot associated with the given sequence number
 * 'seqNr'. Copy 'len' bytes to '*dst' on success.
 *
 * returns:
 *  0	- a collision or overwrite was detected
 *  len - the data integrity was good and all data was copied
 *  ~0	- the destination buffer is bigger than the data slot
 *	  (sets 'errno' to ENOMEM also)
 */
static int
read_next_shm1_slot(
	shmunit_t * const up,
	shm1_data_t *   into)
{
	volatile shm1_head_t *	head;
	volatile u_int32 *	slot;
	u_int32			seqnum_wr, seqnum_rd;
	u_int32			guard1, guard2;

	/* first check if there is anything to read at all... */
	head = get_shm1_head_ptr(&up->smem);
	if (!head)
		return FALSE;
	
	/* fetch read & write sequence numbers
	 * Since there is a strong data dependency, we need no barriers
	 * here -- the volatile declaration enforces the access order
	 * on compiler level, the data dependency on CPU level.
	 */
	seqnum_wr = head->seqnum_wr;
	seqnum_rd = head->seqnum_rd;
	if ((seqnum_wr - seqnum_rd) > SHM1_SLOT_COUNT)
		seqnum_rd = seqnum_wr - SHM1_SLOT_COUNT;

  again:
	printf("\n\nread_next_shm1_slot: init: rd=%u(@%ld), wr=%u(@%ld)\n",
	       seqnum_rd, poffs(&up->smem, &head->seqnum_rd),
	       seqnum_wr, poffs(&up->smem, &head->seqnum_wr));
	/* if there is no more data, just bail out */
	if (seqnum_rd == seqnum_wr)
		return FALSE;
		
	/* read data & guards
	 *
	 * We would need aquisition barriers here, but that's hard to
	 * do. We use atomic_set() in combination with volatile
	 * declarations to achieve the desired effect: 'atomic_set()'
	 * has a full barrier, so we use it to set a local (per se
	 * non-volatile) value from a volatile source, purely for the
	 * barrier side effect.
	 */
	slot = get_shm1_data_ptr(&up->smem, ++seqnum_rd);
	atomic_set(&guard2, slot[SHM1_SLOT_SIZE-1]);
	*into = *(volatile shm1_data_t*)(slot+1);
	atomic_set(&guard1, slot[0]);

	/* If the guards do match each other, the slot is in
	 * transit. Quit here.
	 */
	if (guard1 != guard2)
		return FALSE;
	
	/* If the guards do match the serial number, the record was
	 * already overwritten. Try next.
	 */
	if (guard1 != seqnum_rd) {
		up->clash++;
		goto again;
	}
	
	/* update the head and claim success */
	atomic_set(&head->seqnum_rd, seqnum_rd);
	return TRUE;
}


static void
refclock_vprint_lastcode(
	refclockproc_t * pp ,
	const char *     fmt,
	va_list	         va )
{
	pp->lencode = (u_short)vsnprintf(
		pp->a_lastcode, sizeof(pp->a_lastcode), fmt, va);
	if (pp->lencode >= sizeof(pp->a_lastcode)) {
		pp->a_lastcode[sizeof(pp->a_lastcode)-1] = '\0';
		pp->lencode = sizeof(pp->a_lastcode) - 1;
	}
}

static void
refclock_print_lastcode(
	refclockproc_t * pp ,
	const char *     fmt,
	...                 )
{
	va_list va;
	va_start(va, fmt);
	refclock_vprint_lastcode(pp, fmt, va);
	va_end(va);
}

static int
fmt_lfp_frac(
	char *  buf     ,
	size_t  blen    ,
	u_int32 fraction,
	size_t  digits  ,
	size_t  dotpos  )
{
	size_t need;
	char * cp;
	l_fp val, tmp;
	int carry;

	/* clamp maximum precision to 9 digits; we can't get more than
	 * that from a 32-bit fraction.
	 */
	if (digits > 9)
		digits = 9;
	/* check how many digits fit into buffer */
	need = digits + (dotpos < blen) + 1;
	if (need > blen) {
		need -= blen;
		if  (digits < need) {
			/* total loss of precision -- bail out */
			if (blen)
				*buf = '\0';
			return 0;
		}
		digits -= need;
	}
	
	/* now start splitting off digits */
	val.l_uf = fraction;
	for (cp = buf; digits; cp++,--digits) {
		if ((size_t)(cp - buf) == dotpos)
			*cp++ ='.';
		val.l_ui = 0;
		/* multiply by then the hard way, not using the multiplier... */
		L_LSHIFT(&val);
		tmp = val;
		L_LSHIFT(&tmp);
		L_LSHIFT(&tmp);
		L_ADD(&val, &tmp);
		/* split off one digit */
		*cp = (char)(val.l_ui + '0');
	}

	/* plug in terminator */
	*cp = '\0';

	/* now do the rounding dance */
	carry = ((~val.l_uf + 1) < val.l_uf);
	while (cp != buf && carry)
		if (*--cp != '.')			
			if ((carry = ((*cp += (char)carry) > '9')) != 0)
				*cp -= 10;

	return carry;
}


/*
 * ---------------------------------------------------------------------
 */

/*
 * ---------------------------------------------------------------------
 * Creation of SHM segments is OS dependent -- separate implementations
 * are required. 
 * ---------------------------------------------------------------------
 */

static int getShmTime(
	shmclk_sys_t * const dp, int mode, int unit,
	size_t length, int public);
static void delShmTime(
	shmclk_sys_t * const dp);

static const char * logmsg = "SHM clock (unit=%u, mode=%u): %s: %m";

/* ================================================================== */
#ifdef SYS_WINNT
/* =====================================================================
 *
 * Win32 shared memory (pagefile backed)
 *
 * =====================================================================
 */

static int
getShmTime(
	shmclk_sys_t * const dsc   ,
	int                  mode  ,
	int                  unit  ,
	size_t               length,
	int                  public)
{
        static const char * names[2] = {
                "Global\\NTP%d" , "Global\\NTP%d.%d"};

	SECURITY_ATTRIBUTES sa, *psa;
	SECURITY_DESCRIPTOR sd;
        BOOL                rc;

	if (dsc->sys_addr) {
		errno = EEXIST;
		return FALSE;
	}

        memset(dsc, 0, sizeof(shmclk_sys_t));
	dsc->sys_name = my_aprintf(names[mode > 0], unit, mode);

	if (public) { /* world access */
                rc = InitializeSecurityDescriptor(
                        &sd, SECURITY_DESCRIPTOR_REVISION);
		if ( !rc ) {
			msyslog(LOG_ERR, logmsg,
                                "InitializeSecurityDescriptor",
                                mode, unit);
			goto fail;
		}
                rc = SetSecurityDescriptorDacl(&sd, 1, 0, 0);
		if ( !rc ) {
			msyslog(LOG_ERR, logmsg,
                                "SetSecurityDescriptorDacl",
                                mode, unit);
			goto fail;
		}
		sa.nLength              = sizeof(sa);
		sa.lpSecurityDescriptor = &sd;
		sa.bInheritHandle       = 0;
		psa = &sa;
	} else {
		psa = NULL;
	}
	dsc->sys_handle = CreateFileMapping(INVALID_HANDLE_VALUE, psa,
                                PAGE_READWRITE, 0, length, dsc->sys_name);
	if (!dsc->sys_handle) { /*error*/
		msyslog(LOG_ERR, logmsg, "CreateFileMapping",
			mode, unit);
		goto fail;
	}

	dsc->sys_addr = MapViewOfFile(dsc->sys_handle,
                                FILE_MAP_WRITE, 0, 0, length);
	if (!dsc->sys_addr) { /*error*/
		msyslog(LOG_ERR, logmsg, "MapViewOfFile",
			mode, unit);
		goto fail;
	}
	dsc->sys_size = length;
	return TRUE;

fail:
        if (dsc->sys_handle)
                CloseHandle(dsc->sys_handle);
        free(dsc->sys_name);
        memset(dsc, 0, sizeof(shmclk_sys_t));
	return FALSE;
}

static void
delShmTime(
	shmclk_sys_t * const dsc)
{
	if (dsc->sys_addr && dsc->sys_size) {
		UnmapViewOfFile(dsc->sys_addr);
		CloseHandle(dsc->sys_handle);
                free(dsc->sys_name);
        }
        memset(dsc, 0, sizeof(shmclk_sys_t));
}

/* ================================================================== */
#elif HAVE_SYS_MMAN_H && HAVE_SHM_OPEN && HAVE_SHM_UNLINK
/* =====================================================================
 *
 * POSIX compatible SHM access
 *
 * =====================================================================
 */

static int
getShmTime(
	shmclk_sys_t * const dsc   ,
	int                  mode  ,
	int                  unit  ,
	size_t               length,
	int                  public)
{
	int operm;
	mode_t fmask;

	/* If already attached, bail out without touching anything */
	if (dsc->sys_addr && dsc->sys_size >= length) {
		errno = EEXIST;
		return TRUE;
	}

	errno = 0;
	/* create or open the memory segment */
	operm          = (public ? 0666 : 0600);
	dsc->sys_name  = my_aprintf("/ntpd-shmclk.%u.%u", unit, mode);
	/*
	 * We have to fiddle our umask, or creating a world-accessible
	 * SHM segment will fail. Another option would be using 'fchmod()',
	 * but that is XOPEN or BSD specific... Using 'chmod()' is bound
	 * to fail, as there is no predictable way to map the SHM name to
	 * a true file system name (if there is one at all!)
	 */
	shm_unlink(dsc->sys_name);
	fmask = umask(0);
	dsc->sys_shmid = shm_open(dsc->sys_name, O_RDWR|O_CREAT|O_EXCL, operm);
	umask(fmask);
	if (dsc->sys_shmid < 0) {
		msyslog(LOG_ERR, logmsg, unit, mode, "shm_open");
		goto fail;
	}

	/* make sure it's the right size for us */
	if (ftruncate(dsc->sys_shmid, length)) {
		msyslog(LOG_ERR, logmsg, unit, mode, "ftruncate");
		goto fail;
	}
	dsc->sys_size = length;

	/* now try to get into our address space */
	dsc->sys_addr = mmap(NULL, length, PROT_READ|PROT_WRITE,
			     MAP_SHARED, dsc->sys_shmid, 0);
	if (dsc->sys_addr == MAP_FAILED) {
		msyslog(LOG_ERR, logmsg, unit, mode, "mmap");
		goto fail;
	}

	return TRUE;

  fail:
	if (dsc->sys_shmid >= 0)
		close(dsc->sys_shmid);
	if (dsc->sys_name)
		shm_unlink(dsc->sys_name);
	free (dsc->sys_name);

	memset(dsc, 0, sizeof(shmclk_sys_t));
	return FALSE;
}

static void
delShmTime(
	shmclk_sys_t * const dsc)
{
	if (dsc->sys_addr && dsc->sys_size) {
		munmap(dsc->sys_addr, dsc->sys_size);
		close(dsc->sys_shmid);
		shm_unlink(dsc->sys_name);
		free (dsc->sys_name);
	}
	memset(dsc, 0, sizeof(shmclk_sys_t));
}

/* ================================================================== */
#else
/* =====================================================================
 *
 * SYSV compatible SHM access
 *
 * =====================================================================
 */

static int
getShmTime(
	shmclk_sys_t * const dsc   ,
	int                  mode  ,
	int                  unit  ,
	size_t               length,
	int                  public)
{
	int ident;
	int omode;

	/* If already attached, bail out without touching anything */
	if (dsc->sys_addr && dsc->sys_size >= length) {
		errno = EEXIST;
		return TRUE;
	}

	/* 0x4e545030 is NTP0.
	 * Big units will give non-ascii but that's OK
	 * as long as everybody does it the same way. 
	 */
	ident = 0x4e545030 + mode*256 + unit;
	omode = IPC_CREAT | (public ? 0666 : 0600);
	
	dsc->sys_shmid = shmget(ident, length, omode); 
	if (dsc->sys_shmid == -1) {
		msyslog(LOG_ERR, logmsg, unit, mode, "shmget");
		goto fail;
	}

	dsc->sys_addr = shmat(dsc->sys_shmid, NULL, 0);
	if (dsc->sys_addr == (void*)-1) {
		msyslog(LOG_ERR, logmsg, unit, mode, "shmat");
		goto fail;
	}
	dsc->sys_size = length;
	return TRUE;

  fail:
	memset(dsc, 0, sizeof(shmclk_sys_t));
	return FALSE;
}

static void
delShmTime(
	shmclk_sys_t * dsc)
{
	if (dsc->sys_addr && dsc->sys_size)
		(void)shmdt((char*)dsc->sys_addr);
	memset(dsc, 0, sizeof(shmclk_sys_t));
}

#endif


/*
 * =====================================================================
 * SHM function handlers
 * =====================================================================
 */

/*
 * ---------------------------------------------------------------------
 * type 0 attach / create
 */
static int
attachTyp0(
	shmunit_t * const up)
{
	static const size_t MemSize = sizeof(shm0_data_t);
	volatile shm0_data_t *	shm0;

	int pub = (up->unit >= 2);

	if (getShmTime(&up->smem, 0, up->unit, MemSize, pub)) {
		shm0 = (volatile shm0_data_t *)up->smem.sys_addr;
		/*
		 * Initialize miscellaneous peer variables
		 */
		shm0->precision = up->prec;
		shm0->valid     = 0;
		shm0->nsamples  = NSAMPLES;
		atomic_barrier();
			
		up->peer->precision = (s_char)shm0->precision;
	}
	return (up->smem.sys_addr != NULL);
}

/*
 * ---------------------------------------------------------------------
 * type 0 detach
 */
static void
detachTyp0(
	shmunit_t * const up)
{
	delShmTime(&up->smem);
}

/*
 * ---------------------------------------------------------------------
 * type 0 processing
 */
static void
processTyp0(
	shmunit_t * const up)
{
	volatile shm0_data_t * shm;

	/*
	 * This is the main routine. It snatches the time from the shm
	 * board and tacks on a local timestamp.
	 *
	 * If we are not attached to SHM, try to map again - this may
	 * succeed if meanwhile somebody has ipcrm'ed the old
	 * (unaccessible) shared mem segment. Complain if that fails.
	 */
	if (!(up->smem.sys_addr || attachTyp0(up))) {
		refclock_report(up->peer, CEVNT_FAULT);
		return;
	}

	/*
	 * we can be sure to have a SHM segment below that point
	 */
	shm = (volatile shm0_data_t *)up->smem.sys_addr;
	if (shm->valid) {
		struct timespec tvr, tvt;
		l_fp            recTime, refTime;
		int             cnt, ok;
		unsigned        cns_new, rns_new;

		switch (shm->mode)
		{
		case 0:
			tvr.tv_sec  = shm->receiveTimeStampSec;
			tvr.tv_nsec = shm->receiveTimeStampUSec * 1000;
			rns_new	    = shm->receiveTimeStampNSec;
			tvt.tv_sec  = shm->clockTimeStampSec;
			tvt.tv_nsec = shm->clockTimeStampUSec * 1000;
			cns_new	    = shm->clockTimeStampNSec;
			ok = 1;
			break;

		case 1:
			cnt = shm->count;
			tvr.tv_sec  = shm->receiveTimeStampSec;
			tvr.tv_nsec = shm->receiveTimeStampUSec * 1000;
			rns_new	    = shm->receiveTimeStampNSec;
			tvt.tv_sec  = shm->clockTimeStampSec;
			tvt.tv_nsec = shm->clockTimeStampUSec * 1000;
			cns_new	    = shm->clockTimeStampNSec;
			ok = (cnt == shm->count);
			break;

		default:
			memset(&tvt, 0, sizeof(tvt));
			memset(&tvr, 0, sizeof(tvr));
                        cns_new = 0;
                        rns_new = 0;
			ok = 0;
			msyslog (LOG_ERR,
				 "SHM: bad mode found in shared memory: %d",
				 shm->mode);
		}
		shm->valid = 0;

		/* Check if the nano-second stamps match.
		 * Since these comparisons are between unsigned
		 * variables they are always well defined, and any
		 * (signed) underflow will turn into very large
		 * unsigned values, well above the 1000 cutoff.
		 *
		 * Note that this assumes that the usec-frcations have
		 * been created by truncation, not rounding!
		 */
		if (  ok
		   && ((cns_new - (unsigned)tvt.tv_nsec) < 1000)
		   && ((rns_new - (unsigned)tvr.tv_nsec) < 1000)) {
			tvt.tv_nsec = cns_new;
			tvr.tv_nsec = rns_new;
		}
		
		if (!ok) {
			refclock_report(up->peer, CEVNT_FAULT);
			msyslog (LOG_NOTICE, "SHM: access clash in shared memory");
			up->clash++;
		} else if (timespec_isdenormal(&tvt) ||
			   timespec_isdenormal(&tvr)) {
			refclock_report(up->peer, CEVNT_BADTIME);
			up->bad++;
		} else {
			recTime = tspec_stamp_to_lfp(tvr);
			refTime = tspec_stamp_to_lfp(tvt);
			up->prec = (s_char)shm->precision;
			up->leap = (u_char)shm->leap;
			refclock_process_offset(up->proc, refTime, recTime,
						up->proc->fudgetime1);
			up->good++;
		}
	} else {
		up->notready++;
	}
}

/*
 * ---------------------------------------------------------------------
 * type 1 attach / create
 */
#define SHM1_PUBLIC(mode) (((mode) & 128) != 0)

static int
attachTyp1(
	shmunit_t * const up)
{
	static const size_t Typ1MemSize = sizeof(uint32_t) *
	    (SHM1_SLOT_COUNT * SHM1_SLOT_SIZE + SHM1_HEAD_SIZE);
	    
	int pub = SHM1_PUBLIC(up->peer->ttl);

	if (getShmTime(&up->smem, 1, up->unit, Typ1MemSize, pub)) {
		/*
		 * setup ptrs
		 */
		volatile shm1_head_t *	hdr;
		hdr = get_shm1_head_ptr(&up->smem);
		/*
		 * Initialize miscellaneous SHM & peer variables
		 */
		hdr->server_version = NTP_SHM1_VERSION(1,0);
		hdr->seqnum_rd      = hdr->seqnum_wr;
		hdr->head_dwsize    = SHM1_HEAD_SIZE;
		hdr->data_dwsize    = SHM1_SLOT_SIZE;
		hdr->data_slots     = SHM1_SLOT_COUNT;
		atomic_set(&hdr->magic, SHM1_MAGIC);
		up->peer->precision = up->prec;
	}

	return (up->smem.sys_addr != NULL);
}

/*
 * ---------------------------------------------------------------------
 * type 1 detach
 */
static void
detachTyp1(
	shmunit_t * const up)
{
	volatile shm1_head_t *	hdr;
	hdr = get_shm1_head_ptr(&up->smem);
	if (hdr)
		atomic_set(&hdr->magic,0);
	delShmTime(&up->smem);
}

/*
 * ---------------------------------------------------------------------
 * type 1 processing
 */
static int
process_lfp (
	shmunit_t * const         up     ,
	const shm1_data_t * const pbuff  ,
	l_fp * const              reftime)
{
	l_fp fpt, fpr;
	
	fpt.l_ui = (u_int32)pbuff->reftime_secs + JAN_1970;
	fpt.l_uf = pbuff->reftime_frac;
	fpr.l_ui = (u_int32)pbuff->rcvtime_secs + JAN_1970;
	fpr.l_uf = pbuff->rcvtime_frac;
	refclock_process_offset(up->proc, fpt, fpr,
				up->proc->fudgetime1);
	up->good++;
	*reftime = fpt;
	return TRUE;
}

static int
process_ts(
	shmunit_t * const         up     ,
	const shm1_data_t * const pbuff  ,
	l_fp * const              reftime)
{
	struct timespec tvt, tvr;
	l_fp 		fpt, fpr;

	tvt.tv_sec  = (time_t)pbuff->reftime_secs;
	tvt.tv_nsec = pbuff->reftime_frac;
	tvr.tv_sec  = (time_t)pbuff->rcvtime_secs;
	tvr.tv_nsec = pbuff->rcvtime_frac;

	/* final checks */
	if (timespec_isdenormal(&tvt) || timespec_isdenormal(&tvr)) {
		/* inconsistent data */
		refclock_report(up->peer, CEVNT_BADTIME);
		up->bad++;
		return FALSE;
	}
	
	/* get remaining stuff and feed median filter */
	fpt = tspec_stamp_to_lfp(tvt);
	fpr = tspec_stamp_to_lfp(tvr);
	refclock_process_offset(up->proc, fpt, fpr,
				up->proc->fudgetime1);
	up->good++;
	*reftime = fpt;
	return TRUE;
}

static int
process_tv(
	shmunit_t * const         up     ,
	const shm1_data_t * const pbuff  ,
	l_fp * const              reftime)
{
	struct timeval	tvt, tvr;
	l_fp 		fpt, fpr;

	tvt.tv_sec  = (time_t)pbuff->reftime_secs;
	tvt.tv_usec = pbuff->reftime_frac;
	tvr.tv_sec  = (time_t)pbuff->rcvtime_secs;
	tvr.tv_usec = pbuff->rcvtime_frac;

	/* final checks */
	if (timeval_isdenormal(&tvt) || timeval_isdenormal(&tvr)) {
		/* inconsistent data */
		refclock_report(up->peer, CEVNT_BADTIME);
		up->bad++;
		return FALSE;
	}
	
	/* get remaining stuff and feed median filter */
	fpt = tval_stamp_to_lfp(tvt);
	fpr = tval_stamp_to_lfp(tvr);
	refclock_process_offset(up->proc, fpt, fpr,
				up->proc->fudgetime1);
	up->good++;
	*reftime = fpt;
	return TRUE;
}

static void
processTyp1(
	shmunit_t * const up)
{
	/* SHM header access */
	volatile shm1_head_t * smem;

	/* slot buffers and result inicators */
	shm1_data_t	buf; 
	int		clk_hastime, clk_alive;
	time_t          poll_time, diff_time;
	
	/* tally and stats data */
	struct calendar	date;
	char		fbuf[12];
	l_fp		reftime;

	L_CLR(&reftime);

	/* If we are not attached to SHM, try to map again - this may
	 * succeed if meanwhile somebody has ipcrm'ed the old
	 * (unaccessible) SHM segment. Complain if that fails.
	 */
	if (!(up->smem.sys_addr || attachTyp1(up))) {
		DPRINTF(1, ("%s: Typ1 process: cannot attach SHM segment\n",
			    refnumtoa(&up->peer->srcadr)));
		refclock_report(up->peer, CEVNT_FAULT);
		return;
	}

	/*
	 * we can be sure to have a SHM segment below that point -- fetch data!
	 */
	poll_time = time(NULL);	
	smem = get_shm1_head_ptr(&up->smem);

	/* process live signals */
	atomic_inc(&smem->server_livetick);
	clk_alive   = atomic_set(&smem->client_livetick, 0) != 0;
	clk_hastime = 0;

	/* Read axiliary data fields from SHM header */
	up->leap = smem->client_leap;
	if (up->leap > 3)
		up->leap = 0;

	up->prec = smem->client_prec;
	if (up->prec >= 0)
		up->leap = PRECISION;
	
	up->with_pps = (smem->client_pps == 1);
	up->min_poll = smem->client_minpoll;

	/* Read slots until reading fails. If there is a transient
	 * clash in the data, this will resume on the next cycle. A
	 * permanently damaged slot will stop reading it until it is
	 * finally overwritten after SHM1_SLOT_COUNT cycles, when
	 * reading will resume with the slot sequentially following the
	 * damaged one.
	 *
	 * Note tha a malicious client can make sure the clock becomes
	 * effectively useless. But since we're communicating via shared
	 * memory, there's no way to stop this once the client has
	 * access to it...
	 */
	while (read_next_shm1_slot(up, &buf)) {
		clk_alive = 1;

		/* guard against stale time samples */
                if (buf.tsformat != TSF_none) {
                        diff_time = buf.rcvtime_secs - poll_time;
                        if (diff_time < -10 || diff_time > 10)
        			buf.tsformat = TSF_none;
                }

		/* now do the time conversion and feed samples */
		switch (buf.tsformat) {
		case TSF_lfp:
			clk_hastime |= process_lfp(up, &buf, &reftime);
			break;

		case TSF_timespec:
			clk_hastime |= process_ts(up, &buf, &reftime);
			break;

		case TSF_timeval:
			clk_hastime |= process_tv(up, &buf, &reftime);
			break;

		case TSF_none:
		default      :
			break;
		}

	}
	up->notready += !(clk_alive || clk_hastime);

	if (clk_alive || clk_hastime) {
		if (up->with_pps)
			up->peer->flags |=  FLAG_PPS;
		else
			up->peer->flags &= ~FLAG_PPS;
		up->peer->precision = up->prec;
		up->proc->leap      = up->leap;
		up->active          = 10;
	} else if (up->active && !--up->active) {
		up->peer->flags    &= ~FLAG_PPS;
		up->peer->precision = PRECISION;
		up->proc->leap      = LEAP_NOTINSYNC;
		refclock_report(up->peer, CEVNT_FAULT);
	}

	if (clk_hastime) {
		reftime.l_ui += fmt_lfp_frac(fbuf, sizeof(fbuf), reftime.l_uf, 8, 3);
		ntpcal_ntp_to_date(&date, reftime.l_ui, NULL);
		refclock_print_lastcode(up->proc,
			"%04u-%02u-%02u,%02u:%02u:%02u.%s",
			date.year, date.month, date.monthday,
			date.hour, date.minute, date.second,
			fbuf);
		DPRINTF(2, ("\n\n ---> %s SHM typ1 last timestamp: %s\n\n\n",
			refnumtoa(&up->peer->srcadr),
			up->proc->a_lastcode));
	}
}


/*
 * =====================================================================
 * refclock API functions
 * =====================================================================
 */

/*
 * shm_start - attach to shared memory
 */
static int
shm_start(
	int            unit,
	peer_t * const peer)
{
	refclockproc_t * const pp = peer->procptr;
	shmunit_t *            up;

       	up = emalloc_zero(sizeof(shmunit_t));
	up->handlers = &ShmHandlers[peer->ttl % SHMMODES];
	up->unit = unit;
	up->peer = peer;
	up->proc = pp;
	up->leap = LEAP_NOWARNING;
	up->prec = PRECISION;

	pp->leap    = LEAP_NOWARNING;
	pp->unitptr = (caddr_t)up;
	pp->io.clock_recv = noentry;
	pp->io.srcclock = peer;
	pp->io.datalen = 0;
	pp->io.fd = -1;
	pp->clockdesc = DESCRIPTION;

	peer->precision = PRECISION;
	memcpy((char *)&pp->refid, REFID, 4);

	return ((*up->handlers->attach)(up) != 0);
}


/*
 * shm_shutdown - shut down the clock
 */
static void
shm_shutdown(
	int            unit,
	peer_t * const peer)
{
	refclockproc_t * const	pp = peer->procptr;
	shmunit_t * const	up = (shmunit_t *)pp->unitptr;

	if (up) {
		(*up->handlers->detach)(up);
		free(up);
		pp->unitptr = NULL;
	}
}


/*
 * shm_control - handle config changes
 */
static void
shm_control(
	int unit,
	const struct refclockstat * const in_st ,
	struct refclockstat * const       out_st,
	peer_t * const                    peer  )
{
	refclockproc_t * const	pp = peer->procptr;
	shmunit_t * const	up = (shmunit_t *)pp->unitptr;
	dispatchTable		handlers;

	UNUSED_ARG(in_st);
	UNUSED_ARG(out_st);

	/* check for mode changes */
	handlers = &ShmHandlers[peer->ttl % SHMMODES];
	if (handlers != up->handlers) {		
		(*up->handlers->detach)(up);
		up->handlers = handlers;
		(*up->handlers->attach)(up);
	}
}

	
/*
 * shm_poll - called by the transmit procedure
 */
static void
shm_poll(
	int            unit,
	peer_t * const peer)
{
	refclockproc_t * const	pp = peer->procptr;
	shmunit_t * const	up = (shmunit_t *)pp->unitptr;

	pp->polls++;
	/*
	 * Process median filter samples. If none received, declare a
	 * timeout and keep going. I this persists too long, mark clock
	 * as gone.
	 */
	if (pp->coderecv == pp->codeproc) {
		if (up->active)
			refclock_report(peer, CEVNT_TIMEOUT);
	} else {
		pp->lastref = pp->lastrec;
		refclock_receive(peer);
	}
	if (pp->sloppyclockflag & CLK_FLAG4)
		shm_clockstats(unit, peer);
}


/*
 * shm_timer - periodically process the content of the SHM segment
 */
static void
shm_timer(
	int            unit,
	peer_t * const peer)
{
	refclockproc_t * const	pp = peer->procptr;
	shmunit_t * const	up = (shmunit_t *)pp->unitptr;

	if (up) {
		up->ticks++;
		(*up->handlers->process)(up);
	}
}


/*
 * shm_clockstats - dump and reset counters
 */
static void
shm_clockstats(
	int            unit,
	peer_t * const peer)
{
	refclockproc_t * const	pp = peer->procptr;
	shmunit_t * const	up = (shmunit_t *)pp->unitptr;
	char 		logbuf[256];

	snprintf(logbuf, sizeof(logbuf), "%3d %3d %3d %3d %3d",
		 up->ticks, up->good, up->notready, up->bad, up->clash);
	record_clock_stats(&peer->srcadr, logbuf);

	up->ticks = up->good = up->notready = up->bad = up->clash = 0;
}

/*
 * =====================================================================
 *  helpers
 * =====================================================================
 */
static char *
my_aprintf(
	const char *fmt,
	...            )
{
	va_list va;
	size_t blen, flen;
	char  *buff;

	blen = strlen(fmt) & (SIZE_MAX - 0x7f);
	buff = NULL;
	do {
		free(buff);
		va_start(va, fmt);
		blen += 128;
		buff  = emalloc(blen);
		flen  = vsnprintf(buff, blen, fmt, va);
		va_end(va);
	} while (flen >= blen);
	return realloc(buff, flen+1);
}

#else
int refclock_shm_bs;
#endif /* REFCLOCK */
